/*
 * Decompiled with CFR 0.152.
 */
package org.primeframework.mvc.cors;

import io.fusionauth.http.HTTPMethod;
import io.fusionauth.http.server.HTTPRequest;
import io.fusionauth.http.server.HTTPResponse;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import org.primeframework.mvc.cors.CORSDebugger;
import org.primeframework.mvc.workflow.WorkflowChain;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CORSFilter {
    private static final Collection<HTTPMethod> ComplexHTTPMethods = Set.of(HTTPMethod.PATCH, HTTPMethod.PUT, HTTPMethod.DELETE, HTTPMethod.TRACE, HTTPMethod.CONNECT);
    private static final Collection<String> SimpleHTTPRequestContentTypes = Set.of("application/x-www-form-urlencoded", "multipart/form-data", "text/plain");
    private static final Logger logger = LoggerFactory.getLogger(CORSFilter.class);
    private final Collection<String> allowedHTTPHeaders = new LinkedHashSet<String>();
    private final Collection<String> allowedHTTPHeadersOriginal = new LinkedHashSet<String>();
    private final Collection<HTTPMethod> allowedHTTPMethods = new LinkedHashSet<HTTPMethod>();
    private final Collection<String> allowedOrigins = new LinkedHashSet<String>();
    private final Collection<String> exposedHeaders = new LinkedHashSet<String>();
    private boolean anyOriginAllowed;
    private boolean debug;
    private CORSDebugger debugger;
    private Predicate<String> excludeURIPredicate;
    private Pattern excludedPathPattern;
    private Predicate<String> includeURIPredicate;
    private Pattern includedPathPattern;
    private long preflightMaxAge;
    private boolean supportsCredentials;

    public void doFilter(HTTPRequest request, HTTPResponse response, WorkflowChain workflowChain) throws IOException {
        String origin = request.getHeader("Origin");
        if (origin != null && this.isSameOrigin(origin, request)) {
            workflowChain.continueWorkflow();
            return;
        }
        CORSRequestType requestType = this.checkRequestType(request, origin);
        String requestURI = request.getPath();
        if (this.excludedRequestURI(requestURI)) {
            if (requestType == CORSRequestType.PRE_FLIGHT) {
                this.handleInvalidCORS(request, response, InvalidCORSReason.PreFlightUnexpected, requestURI);
                return;
            }
            requestType = CORSRequestType.NOT_CORS;
        }
        switch (requestType.ordinal()) {
            case 0: 
            case 1: {
                this.handleSimpleCORS(request, response, workflowChain);
                break;
            }
            case 2: {
                this.handlePreflightCORS(request, response);
                break;
            }
            case 3: {
                workflowChain.continueWorkflow();
                break;
            }
            default: {
                this.handleInvalidCORS(request, response, InvalidCORSReason.UnhandledCORSRequestType, (Object)requestType);
            }
        }
    }

    public CORSFilter withAllowCredentials(boolean allow) {
        this.supportsCredentials = allow;
        return this;
    }

    public CORSFilter withAllowedHTTPHeaders(List<String> headers) {
        if (headers != null) {
            this.allowedHTTPHeaders.clear();
            this.allowedHTTPHeadersOriginal.clear();
            for (String header : headers) {
                this.allowedHTTPHeaders.add(header.toLowerCase());
                this.allowedHTTPHeadersOriginal.add(header);
            }
        }
        return this;
    }

    public CORSFilter withAllowedHTTPMethods(List<HTTPMethod> methods) {
        if (methods != null) {
            this.allowedHTTPMethods.clear();
            this.allowedHTTPMethods.addAll(methods);
        }
        return this;
    }

    public CORSFilter withAllowedOrigins(List<URI> origins) {
        if (origins != null) {
            if (origins.contains(URI.create("*"))) {
                this.anyOriginAllowed = true;
            } else {
                this.anyOriginAllowed = false;
                this.allowedOrigins.clear();
                origins.forEach(o -> this.allowedOrigins.add(o.toString()));
            }
        }
        return this;
    }

    public CORSFilter withDebugEnabled(boolean debug) {
        this.debug = debug;
        return this;
    }

    public CORSFilter withDebugger(CORSDebugger debugger) {
        this.debugger = debugger;
        return this;
    }

    public CORSFilter withExcludeURIPredicate(Predicate<String> excludeURIPredicate) {
        this.excludeURIPredicate = excludeURIPredicate;
        return this;
    }

    public CORSFilter withExcludedPathPattern(Pattern pattern) {
        this.excludedPathPattern = pattern;
        return this;
    }

    public CORSFilter withExposedHeaders(List<String> headers) {
        if (headers != null) {
            this.exposedHeaders.clear();
            this.exposedHeaders.addAll(headers);
        }
        return this;
    }

    public CORSFilter withIncludeURIPredicate(Predicate<String> includeURIPredicate) {
        this.includeURIPredicate = includeURIPredicate;
        return this;
    }

    public CORSFilter withIncludedPathPattern(Pattern pattern) {
        this.includedPathPattern = pattern;
        return this;
    }

    public CORSFilter withPreflightMaxAge(int maxAge) {
        this.preflightMaxAge = maxAge;
        return this;
    }

    private CORSRequestType checkRequestType(HTTPRequest request, String origin) {
        if (request == null) {
            throw new IllegalArgumentException("HttpServletRequest object is null");
        }
        if (origin == null) {
            return CORSRequestType.NOT_CORS;
        }
        if (origin.isBlank() || !this.isValidOrigin(origin)) {
            return CORSRequestType.INVALID_CORS;
        }
        CORSRequestType requestType = CORSRequestType.INVALID_CORS;
        HTTPMethod method = request.getMethod();
        if (method != null) {
            if (HTTPMethod.OPTIONS.is(method)) {
                String accessControlRequestMethodHeader = request.getHeader("Access-Control-Request-Method");
                if (accessControlRequestMethodHeader != null && !accessControlRequestMethodHeader.isBlank()) {
                    requestType = CORSRequestType.PRE_FLIGHT;
                } else if (accessControlRequestMethodHeader == null) {
                    requestType = CORSRequestType.ACTUAL;
                }
            } else if (HTTPMethod.GET.is(method) || HTTPMethod.HEAD.is(method)) {
                requestType = CORSRequestType.SIMPLE;
            } else if (HTTPMethod.POST.is(method)) {
                String contentType = request.getContentType();
                if (contentType != null) {
                    requestType = SimpleHTTPRequestContentTypes.contains(contentType = contentType.toLowerCase().trim()) ? CORSRequestType.SIMPLE : CORSRequestType.ACTUAL;
                }
            } else if (ComplexHTTPMethods.contains(method)) {
                requestType = CORSRequestType.ACTUAL;
            }
        }
        return requestType;
    }

    private boolean excludedRequestURI(String requestURI) {
        if (this.excludedPathPattern != null) {
            return this.excludedPathPattern.matcher(requestURI).find();
        }
        if (this.includedPathPattern != null) {
            return !this.includedPathPattern.matcher(requestURI).find();
        }
        if (this.includeURIPredicate != null) {
            return !this.includeURIPredicate.test(requestURI);
        }
        if (this.excludeURIPredicate != null) {
            return this.excludeURIPredicate.test(requestURI);
        }
        return false;
    }

    private void handleInvalidCORS(HTTPRequest request, HTTPResponse response, InvalidCORSReason reason, Object reasonValue) {
        if (logger.isDebugEnabled() || this.debug) {
            this.logRequest(request, reason, reasonValue);
        }
        response.setContentType("text/plain");
        response.setStatus(403);
    }

    private void handlePreflightCORS(HTTPRequest request, HTTPResponse response) {
        HTTPMethod accessControlRequestMethod;
        String origin = request.getHeader("Origin");
        if (!this.isOriginAllowed(origin)) {
            this.handleInvalidCORS(request, response, InvalidCORSReason.PreFlightOriginNotAllowed, origin);
            return;
        }
        String accessControlRequestMethodValue = request.getHeader("Access-Control-Request-Method");
        HTTPMethod hTTPMethod = accessControlRequestMethod = accessControlRequestMethodValue != null ? HTTPMethod.of((String)accessControlRequestMethodValue.trim()) : null;
        if (accessControlRequestMethod == null) {
            this.handleInvalidCORS(request, response, InvalidCORSReason.PreFlightMethodNotRecognized, null);
            return;
        }
        String accessControlRequestHeadersHeader = request.getHeader("Access-Control-Request-Headers");
        LinkedList<String> accessControlRequestHeaders = new LinkedList<String>();
        if (accessControlRequestHeadersHeader != null && !accessControlRequestHeadersHeader.trim().isEmpty()) {
            String[] headers;
            for (String header : headers = accessControlRequestHeadersHeader.trim().split(",")) {
                accessControlRequestHeaders.add(header.trim().toLowerCase());
            }
        }
        if (!this.allowedHTTPMethods.contains(accessControlRequestMethod)) {
            this.handleInvalidCORS(request, response, InvalidCORSReason.PreFlightMethodNotAllowed, accessControlRequestMethod);
            return;
        }
        if (!accessControlRequestHeaders.isEmpty()) {
            for (String header : accessControlRequestHeaders) {
                if (this.allowedHTTPHeaders.contains(header)) continue;
                this.handleInvalidCORS(request, response, InvalidCORSReason.PreFlightHeaderNotAllowed, header);
                return;
            }
        }
        if (this.supportsCredentials) {
            response.addHeader("Access-Control-Allow-Origin", origin);
            response.addHeader("Access-Control-Allow-Credentials", "true");
            response.addHeader("Vary", "Origin");
        } else if (this.anyOriginAllowed) {
            response.addHeader("Access-Control-Allow-Origin", "*");
        } else {
            response.addHeader("Access-Control-Allow-Origin", origin);
            response.addHeader("Vary", "Origin");
        }
        if (this.preflightMaxAge > 0L) {
            response.addHeader("Access-Control-Max-Age", String.valueOf(this.preflightMaxAge));
        }
        response.addHeader("Access-Control-Allow-Methods", accessControlRequestMethod.toString());
        if (!this.allowedHTTPHeaders.isEmpty()) {
            response.addHeader("Access-Control-Allow-Headers", String.join((CharSequence)",", this.allowedHTTPHeadersOriginal));
        }
        response.setStatus(204);
    }

    private void handleSimpleCORS(HTTPRequest request, HTTPResponse response, WorkflowChain workflowChain) throws IOException {
        String origin = request.getHeader("Origin");
        HTTPMethod method = request.getMethod();
        if (!this.isOriginAllowed(origin)) {
            this.handleInvalidCORS(request, response, InvalidCORSReason.SimpleOriginNotAllowed, origin);
            return;
        }
        if (!this.allowedHTTPMethods.contains(method)) {
            this.handleInvalidCORS(request, response, InvalidCORSReason.SimpleMethodNotAllowed, method);
            return;
        }
        if (this.anyOriginAllowed && !this.supportsCredentials) {
            response.addHeader("Access-Control-Allow-Origin", "*");
        } else {
            response.addHeader("Access-Control-Allow-Origin", origin);
            response.addHeader("Vary", "Origin");
        }
        if (this.supportsCredentials) {
            response.addHeader("Access-Control-Allow-Credentials", "true");
        }
        if (this.exposedHeaders.size() > 0) {
            String exposedHeadersString = String.join((CharSequence)",", this.exposedHeaders);
            response.addHeader("Access-Control-Expose-Headers", exposedHeadersString);
        }
        workflowChain.continueWorkflow();
    }

    private boolean isOriginAllowed(String origin) {
        if (this.anyOriginAllowed) {
            return true;
        }
        return this.allowedOrigins.contains(origin);
    }

    private boolean isSameOrigin(String origin, HTTPRequest request) {
        if ("null".equals(origin)) {
            return false;
        }
        if (origin.startsWith("file://")) {
            return false;
        }
        try {
            URI uri = URI.create(request.getBaseURL());
            URI originURI = URI.create(origin);
            return uri.getScheme().equalsIgnoreCase(originURI.getScheme()) & uri.getPort() == originURI.getPort() && uri.getHost().equalsIgnoreCase(originURI.getHost());
        }
        catch (Exception exception) {
            return false;
        }
    }

    private boolean isValidOrigin(String origin) {
        URI originURI;
        if (origin.contains("%")) {
            return false;
        }
        if ("null".equals(origin)) {
            return true;
        }
        if (origin.startsWith("file://")) {
            return true;
        }
        try {
            originURI = new URI(origin);
        }
        catch (URISyntaxException e) {
            return false;
        }
        return originURI.getScheme() != null;
    }

    private void logRequest(HTTPRequest request, InvalidCORSReason reason, Object reasonValue) {
        if (this.debugger == null) {
            return;
        }
        String message = switch (reason.ordinal()) {
            default -> throw new MatchException(null, null);
            case 1 -> "Invalid request. Not expecting a preflight request from URI [" + String.valueOf(reasonValue) + "].";
            case 6 -> "Invalid Simple CORS request. HTTP method not allowed. [" + String.valueOf(reasonValue) + "]";
            case 5 -> "Invalid Simple CORS request. Origin not allowed. [" + String.valueOf(reasonValue) + "]";
            case 2 -> "Invalid CORS pre-flight request. HTTP header not allowed. [" + String.valueOf(reasonValue) + "]";
            case 3 -> "Invalid CORS pre-flight request. HTTP method not allowed. [" + String.valueOf(reasonValue) + "]";
            case 4 -> "Invalid CORS pre-flight request. HTTP method not recognized. [" + String.valueOf(reasonValue) + "]";
            case 0 -> "Invalid CORS pre-flight request. Origin not allowed. [" + String.valueOf(reasonValue) + "]";
            case 7 -> "Invalid request. Unhandled CORS request type [" + String.valueOf(reasonValue) + "].";
        };
        this.debugger.disableTimestamp().log(message, new Object[0]).log("", new Object[0]).log("Base URI: %s", URI.create(request.getBaseURL())).log("HTTP Method: %s", request.getMethod()).log("URI: %s", request.getPath()).log("", new Object[0]).log("Content-Type header: %s", request.getHeader("Content-Type")).log("Host header: %s", request.getHeader("Host")).log("Origin header: %s", request.getHeader("Origin")).log("Referer header: %s", request.getHeader("Referer")).log("", new Object[0]).log("Remote host: %s", request.getHost()).log("IP address: %s", request.getIPAddress()).log("", new Object[0]).log("Header names: %s", String.join((CharSequence)",", request.getHeaders().keySet()));
        if (request.getHeader("Content-Type") == null && reason == InvalidCORSReason.UnhandledCORSRequestType) {
            this.debugger.log("", new Object[0]).log("You are missing the Content-Type header during a POST request. This is an invalid CORS request and is the likely root cause of this failure.", new Object[0]);
        }
        this.debugger.log("", new Object[0]).log("Return HTTP Status code 403.", new Object[0]);
        if (logger.isDebugEnabled()) {
            logger.debug(this.debugger.toString());
        }
        if (this.debug) {
            this.debugger.done();
        }
    }

    public static enum CORSRequestType {
        SIMPLE,
        ACTUAL,
        PRE_FLIGHT,
        NOT_CORS,
        INVALID_CORS;

    }

    static enum InvalidCORSReason {
        PreFlightOriginNotAllowed,
        PreFlightUnexpected,
        PreFlightHeaderNotAllowed,
        PreFlightMethodNotAllowed,
        PreFlightMethodNotRecognized,
        SimpleOriginNotAllowed,
        SimpleMethodNotAllowed,
        UnhandledCORSRequestType;

    }
}

