"use strict";
/*
* Copyright (c) 2019-2025, FusionAuth, All Rights Reserved
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
* either express or implied. See the License for the specific
* language governing permissions and limitations under the License.
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebhookEventResult = exports.WebhookAttemptResult = exports.WebAuthnWorkflow = exports.VerificationStrategy = exports.UserVerificationRequirement = exports.UserState = exports.UserActionPhase = exports.UnverifiedBehavior = exports.UnknownScopePolicy = exports.TransactionType = exports.TokenType = exports.ThemeType = exports.UniqueUsernameStrategy = exports.SystemTrustedProxyConfigurationPolicy = exports.SteamAPIMode = exports.Sort = exports.SendSetPasswordIdentityType = exports.SecureGeneratorType = exports.SAMLv2DestinationAssertionPolicy = exports.ResidentKeyRequirement = exports.RefreshTokenUsagePolicy = exports.RefreshTokenExpirationPolicy = exports.ReactorFeatureStatus = exports.PublicKeyCredentialType = exports.ProofKeyForCodeExchangePolicy = exports.PasswordlessStrategy = exports.BreachMatchMode = exports.BreachAction = exports.ObjectState = exports.Oauth2AuthorizedURLValidationPolicy = exports.OAuthScopeHandlingPolicy = exports.OAuthScopeConsentMode = exports.OAuthErrorType = exports.OAuthErrorReason = exports.OAuthApplicationRelationship = exports.MultiFactorLoginPolicy = exports.MessengerType = exports.MessageType = exports.LogoutBehavior = exports.LambdaType = exports.LambdaEngineType = exports.LDAPSecurityMethod = exports.KeyUse = exports.KeyType = exports.KeyAlgorithm = exports.IdentityVerifiedReason = exports.IdentityProviderType = exports.ClientAuthenticationMethod = exports.IdentityProviderLoginMethod = exports.IdentityProviderLinkingStrategy = exports.IPAccessControlEntryAction = exports.HTTPMethod = exports.GrantType = exports.FormType = exports.FormFieldAdminPolicy = exports.FormDataType = exports.FormControl = exports.FamilyRole = exports.ExpiryUnit = exports.EventType = exports.EventLogType = exports.EmailSecurityType = exports.DeviceType = exports.CoseKeyType = exports.CoseEllipticCurve = exports.CoseAlgorithmIdentifier = exports.ContentStatus = exports.ConsentStatus = exports.ConnectorType = exports.ClientAuthenticationPolicy = exports.ChangePasswordReason = exports.CaptchaMethod = exports.CanonicalizationMethod = exports.BreachedPasswordStatus = exports.TOTPAlgorithm = exports.AuthenticatorAttachmentPreference = exports.AuthenticatorAttachment = exports.AuthenticationThreats = exports.AttestationType = exports.AttestationConveyancePreference = exports.ApplicationMultiFactorTrustPolicy = exports.XMLSignatureLocation = exports.SAMLLogoutBehavior = exports.RegistrationType = exports.LoginIdType = exports.Algorithm = exports.FusionAuthClient = void 0;
const DefaultRESTClientBuilder_1 = require("./DefaultRESTClientBuilder");
const url_1 = require("url");
class FusionAuthClient {
    constructor(apiKey, host, tenantId) {
        this.apiKey = apiKey;
        this.host = host;
        this.tenantId = tenantId;
        this.clientBuilder = new DefaultRESTClientBuilder_1.default();
    }
    /**
     * Sets the tenant id, that will be included in the X-FusionAuth-TenantId header.
     *
     * @param {string | null} tenantId The value of the X-FusionAuth-TenantId header.
     * @returns {FusionAuthClient}
     */
    setTenantId(tenantId) {
        this.tenantId = tenantId;
        return this;
    }
    /**
     * Sets whether and how cookies will be sent with each request.
     *
     * @param value The value that indicates whether and how cookies will be sent.
     * @returns {FusionAuthClient}
     */
    setRequestCredentials(value) {
        this.credentials = value;
        return this;
    }
    /**
     * Takes an action on a user. The user being actioned is called the "actionee" and the user taking the action is called the
     * "actioner". Both user ids are required in the request object.
     *
     * @param {ActionRequest} request The action request that includes all the information about the action being taken including
     *    the Id of the action, any options and the duration (if applicable).
     * @returns {Promise<ClientResponse<ActionResponse>>}
     */
    actionUser(request) {
        return this.start()
            .withUri('/api/user/action')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Activates the FusionAuth Reactor using a license Id and optionally a license text (for air-gapped deployments)
     *
     * @param {ReactorRequest} request An optional request that contains the license text to activate Reactor (useful for air-gap deployments of FusionAuth).
     * @returns {Promise<ClientResponse<void>>}
     */
    activateReactor(request) {
        return this.start()
            .withUri('/api/reactor')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Adds a user to an existing family. The family Id must be specified.
     *
     * @param {UUID} familyId The Id of the family.
     * @param {FamilyRequest} request The request object that contains all the information used to determine which user to add to the family.
     * @returns {Promise<ClientResponse<FamilyResponse>>}
     */
    addUserToFamily(familyId, request) {
        return this.start()
            .withUri('/api/user/family')
            .withUriSegment(familyId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Approve a device grant.
     *
     * @param {string} client_id (Optional) The unique client identifier. The client Id is the Id of the FusionAuth Application in which you are attempting to authenticate.
     * @param {string} client_secret (Optional) The client secret. This value will be required if client authentication is enabled.
     * @param {string} token The access token used to identify the user.
     * @param {string} user_code The end-user verification code.
     * @returns {Promise<ClientResponse<DeviceApprovalResponse>>}
     */
    approveDevice(client_id, client_secret, token, user_code) {
        let body = new url_1.URLSearchParams();
        body.append('client_id', client_id);
        body.append('client_secret', client_secret);
        body.append('token', token);
        body.append('user_code', user_code);
        return this.start()
            .withUri('/oauth2/device/approve')
            .withFormData(body)
            .withMethod("POST")
            .go();
    }
    /**
     * Cancels the user action.
     *
     * @param {UUID} actionId The action Id of the action to cancel.
     * @param {ActionRequest} request The action request that contains the information about the cancellation.
     * @returns {Promise<ClientResponse<ActionResponse>>}
     */
    cancelAction(actionId, request) {
        return this.start()
            .withUri('/api/user/action')
            .withUriSegment(actionId)
            .withJSONBody(request)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Changes a user's password using the change password Id. This usually occurs after an email has been sent to the user
     * and they clicked on a link to reset their password.
     *
     * As of version 1.32.2, prefer sending the changePasswordId in the request body. To do this, omit the first parameter, and set
     * the value in the request body.
     *
     * @param {string} changePasswordId The change password Id used to find the user. This value is generated by FusionAuth once the change password workflow has been initiated.
     * @param {ChangePasswordRequest} request The change password request that contains all the information used to change the password.
     * @returns {Promise<ClientResponse<ChangePasswordResponse>>}
     */
    changePassword(changePasswordId, request) {
        return this.startAnonymous()
            .withUri('/api/user/change-password')
            .withUriSegment(changePasswordId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Changes a user's password using their access token (JWT) instead of the changePasswordId
     * A common use case for this method will be if you want to allow the user to change their own password.
     *
     * Remember to send refreshToken in the request body if you want to get a new refresh token when login using the returned oneTimePassword.
     *
     * @param {string} encodedJWT The encoded JWT (access token).
     * @param {ChangePasswordRequest} request The change password request that contains all the information used to change the password.
     * @returns {Promise<ClientResponse<ChangePasswordResponse>>}
     *
     * @deprecated This method has been renamed to changePasswordUsingJWT, use that method instead.
     */
    changePasswordByJWT(encodedJWT, request) {
        return this.startAnonymous()
            .withUri('/api/user/change-password')
            .withAuthorization('Bearer ' + encodedJWT)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Changes a user's password using their identity (loginId and password). Using a loginId instead of the changePasswordId
     * bypasses the email verification and allows a password to be changed directly without first calling the #forgotPassword
     * method.
     *
     * @param {ChangePasswordRequest} request The change password request that contains all the information used to change the password.
     * @returns {Promise<ClientResponse<void>>}
     */
    changePasswordByIdentity(request) {
        return this.start()
            .withUri('/api/user/change-password')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Changes a user's password using their access token (JWT) instead of the changePasswordId
     * A common use case for this method will be if you want to allow the user to change their own password.
     *
     * Remember to send refreshToken in the request body if you want to get a new refresh token when login using the returned oneTimePassword.
     *
     * @param {string} encodedJWT The encoded JWT (access token).
     * @param {ChangePasswordRequest} request The change password request that contains all the information used to change the password.
     * @returns {Promise<ClientResponse<ChangePasswordResponse>>}
     */
    changePasswordUsingJWT(encodedJWT, request) {
        return this.startAnonymous()
            .withUri('/api/user/change-password')
            .withAuthorization('Bearer ' + encodedJWT)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Check to see if the user must obtain a Trust Token Id in order to complete a change password request.
     * When a user has enabled Two-Factor authentication, before you are allowed to use the Change Password API to change
     * your password, you must obtain a Trust Token by completing a Two-Factor Step-Up authentication.
     *
     * An HTTP status code of 400 with a general error code of [TrustTokenRequired] indicates that a Trust Token is required to make a POST request to this API.
     *
     * @param {string} changePasswordId The change password Id used to find the user. This value is generated by FusionAuth once the change password workflow has been initiated.
     * @returns {Promise<ClientResponse<void>>}
     */
    checkChangePasswordUsingId(changePasswordId) {
        return this.startAnonymous()
            .withUri('/api/user/change-password')
            .withUriSegment(changePasswordId)
            .withMethod("GET")
            .go();
    }
    /**
     * Check to see if the user must obtain a Trust Token Id in order to complete a change password request.
     * When a user has enabled Two-Factor authentication, before you are allowed to use the Change Password API to change
     * your password, you must obtain a Trust Token by completing a Two-Factor Step-Up authentication.
     *
     * An HTTP status code of 400 with a general error code of [TrustTokenRequired] indicates that a Trust Token is required to make a POST request to this API.
     *
     * @param {string} encodedJWT The encoded JWT (access token).
     * @returns {Promise<ClientResponse<void>>}
     */
    checkChangePasswordUsingJWT(encodedJWT) {
        return this.startAnonymous()
            .withUri('/api/user/change-password')
            .withAuthorization('Bearer ' + encodedJWT)
            .withMethod("GET")
            .go();
    }
    /**
     * Check to see if the user must obtain a Trust Request Id in order to complete a change password request.
     * When a user has enabled Two-Factor authentication, before you are allowed to use the Change Password API to change
     * your password, you must obtain a Trust Request Id by completing a Two-Factor Step-Up authentication.
     *
     * An HTTP status code of 400 with a general error code of [TrustTokenRequired] indicates that a Trust Token is required to make a POST request to this API.
     *
     * @param {string} loginId The loginId (email or username) of the User that you intend to change the password for.
     * @returns {Promise<ClientResponse<void>>}
     */
    checkChangePasswordUsingLoginId(loginId) {
        return this.start()
            .withUri('/api/user/change-password')
            .withParameter('loginId', loginId)
            .withMethod("GET")
            .go();
    }
    /**
     * Check to see if the user must obtain a Trust Request Id in order to complete a change password request.
     * When a user has enabled Two-Factor authentication, before you are allowed to use the Change Password API to change
     * your password, you must obtain a Trust Request Id by completing a Two-Factor Step-Up authentication.
     *
     * An HTTP status code of 400 with a general error code of [TrustTokenRequired] indicates that a Trust Token is required to make a POST request to this API.
     *
     * @param {string} loginId The loginId of the User that you intend to change the password for.
     * @param {Array<String>} loginIdTypes The identity types that FusionAuth will compare the loginId to.
     * @returns {Promise<ClientResponse<void>>}
     */
    checkChangePasswordUsingLoginIdAndLoginIdTypes(loginId, loginIdTypes) {
        return this.start()
            .withUri('/api/user/change-password')
            .withParameter('loginId', loginId)
            .withParameter('loginIdTypes', loginIdTypes)
            .withMethod("GET")
            .go();
    }
    /**
     * Make a Client Credentials grant request to obtain an access token.
     *
     * @param {string} client_id (Optional) The client identifier. The client Id is the Id of the FusionAuth Entity in which you are attempting to authenticate.
     *    This parameter is optional when Basic Authorization is used to authenticate this request.
     * @param {string} client_secret (Optional) The client secret used to authenticate this request.
     *    This parameter is optional when Basic Authorization is used to authenticate this request.
     * @param {string} scope (Optional) This parameter is used to indicate which target entity you are requesting access. To request access to an entity, use the format target-entity:&lt;target-entity-id&gt;:&lt;roles&gt;. Roles are an optional comma separated list.
     * @returns {Promise<ClientResponse<AccessToken>>}
     */
    clientCredentialsGrant(client_id, client_secret, scope) {
        let body = new url_1.URLSearchParams();
        body.append('client_id', client_id);
        body.append('client_secret', client_secret);
        body.append('grant_type', 'client_credentials');
        body.append('scope', scope);
        return this.startAnonymous()
            .withUri('/oauth2/token')
            .withFormData(body)
            .withMethod("POST")
            .go();
    }
    /**
     * Adds a comment to the user's account.
     *
     * @param {UserCommentRequest} request The request object that contains all the information used to create the user comment.
     * @returns {Promise<ClientResponse<UserCommentResponse>>}
     */
    commentOnUser(request) {
        return this.start()
            .withUri('/api/user/comment')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Completes verification of an identity using verification codes from the Verify Start API.
     *
     * @param {VerifyCompleteRequest} request The identity verify complete request that contains all the information used to verify the identity.
     * @returns {Promise<ClientResponse<VerifyCompleteResponse>>}
     */
    completeVerifyIdentity(request) {
        return this.start()
            .withUri('/api/identity/verify/complete')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Complete a WebAuthn authentication ceremony by validating the signature against the previously generated challenge without logging the user in
     *
     * @param {WebAuthnLoginRequest} request An object containing data necessary for completing the authentication ceremony
     * @returns {Promise<ClientResponse<WebAuthnAssertResponse>>}
     */
    completeWebAuthnAssertion(request) {
        return this.startAnonymous()
            .withUri('/api/webauthn/assert')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Complete a WebAuthn authentication ceremony by validating the signature against the previously generated challenge and then login the user in
     *
     * @param {WebAuthnLoginRequest} request An object containing data necessary for completing the authentication ceremony
     * @returns {Promise<ClientResponse<LoginResponse>>}
     */
    completeWebAuthnLogin(request) {
        return this.startAnonymous()
            .withUri('/api/webauthn/login')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Complete a WebAuthn registration ceremony by validating the client request and saving the new credential
     *
     * @param {WebAuthnRegisterCompleteRequest} request An object containing data necessary for completing the registration ceremony
     * @returns {Promise<ClientResponse<WebAuthnRegisterCompleteResponse>>}
     */
    completeWebAuthnRegistration(request) {
        return this.start()
            .withUri('/api/webauthn/register/complete')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates an API key. You can optionally specify a unique Id for the key, if not provided one will be generated.
     * an API key can only be created with equal or lesser authority. An API key cannot create another API key unless it is granted
     * to that API key.
     *
     * If an API key is locked to a tenant, it can only create API Keys for that same tenant.
     *
     * @param {UUID} keyId (Optional) The unique Id of the API key. If not provided a secure random Id will be generated.
     * @param {APIKeyRequest} request The request object that contains all the information needed to create the APIKey.
     * @returns {Promise<ClientResponse<APIKeyResponse>>}
     */
    createAPIKey(keyId, request) {
        return this.start()
            .withUri('/api/api-key')
            .withUriSegment(keyId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates an application. You can optionally specify an Id for the application, if not provided one will be generated.
     *
     * @param {UUID} applicationId (Optional) The Id to use for the application. If not provided a secure random UUID will be generated.
     * @param {ApplicationRequest} request The request object that contains all the information used to create the application.
     * @returns {Promise<ClientResponse<ApplicationResponse>>}
     */
    createApplication(applicationId, request) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a new role for an application. You must specify the Id of the application you are creating the role for.
     * You can optionally specify an Id for the role inside the ApplicationRole object itself, if not provided one will be generated.
     *
     * @param {UUID} applicationId The Id of the application to create the role on.
     * @param {UUID} roleId (Optional) The Id of the role. If not provided a secure random UUID will be generated.
     * @param {ApplicationRequest} request The request object that contains all the information used to create the application role.
     * @returns {Promise<ClientResponse<ApplicationResponse>>}
     */
    createApplicationRole(applicationId, roleId, request) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withUriSegment("role")
            .withUriSegment(roleId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates an audit log with the message and user name (usually an email). Audit logs should be written anytime you
     * make changes to the FusionAuth database. When using the FusionAuth App web interface, any changes are automatically
     * written to the audit log. However, if you are accessing the API, you must write the audit logs yourself.
     *
     * @param {AuditLogRequest} request The request object that contains all the information used to create the audit log entry.
     * @returns {Promise<ClientResponse<AuditLogResponse>>}
     */
    createAuditLog(request) {
        return this.start()
            .withUri('/api/system/audit-log')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a connector.  You can optionally specify an Id for the connector, if not provided one will be generated.
     *
     * @param {UUID} connectorId (Optional) The Id for the connector. If not provided a secure random UUID will be generated.
     * @param {ConnectorRequest} request The request object that contains all the information used to create the connector.
     * @returns {Promise<ClientResponse<ConnectorResponse>>}
     */
    createConnector(connectorId, request) {
        return this.start()
            .withUri('/api/connector')
            .withUriSegment(connectorId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a user consent type. You can optionally specify an Id for the consent type, if not provided one will be generated.
     *
     * @param {UUID} consentId (Optional) The Id for the consent. If not provided a secure random UUID will be generated.
     * @param {ConsentRequest} request The request object that contains all the information used to create the consent.
     * @returns {Promise<ClientResponse<ConsentResponse>>}
     */
    createConsent(consentId, request) {
        return this.start()
            .withUri('/api/consent')
            .withUriSegment(consentId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates an email template. You can optionally specify an Id for the template, if not provided one will be generated.
     *
     * @param {UUID} emailTemplateId (Optional) The Id for the template. If not provided a secure random UUID will be generated.
     * @param {EmailTemplateRequest} request The request object that contains all the information used to create the email template.
     * @returns {Promise<ClientResponse<EmailTemplateResponse>>}
     */
    createEmailTemplate(emailTemplateId, request) {
        return this.start()
            .withUri('/api/email/template')
            .withUriSegment(emailTemplateId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates an Entity. You can optionally specify an Id for the Entity. If not provided one will be generated.
     *
     * @param {UUID} entityId (Optional) The Id for the Entity. If not provided a secure random UUID will be generated.
     * @param {EntityRequest} request The request object that contains all the information used to create the Entity.
     * @returns {Promise<ClientResponse<EntityResponse>>}
     */
    createEntity(entityId, request) {
        return this.start()
            .withUri('/api/entity')
            .withUriSegment(entityId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a Entity Type. You can optionally specify an Id for the Entity Type, if not provided one will be generated.
     *
     * @param {UUID} entityTypeId (Optional) The Id for the Entity Type. If not provided a secure random UUID will be generated.
     * @param {EntityTypeRequest} request The request object that contains all the information used to create the Entity Type.
     * @returns {Promise<ClientResponse<EntityTypeResponse>>}
     */
    createEntityType(entityTypeId, request) {
        return this.start()
            .withUri('/api/entity/type')
            .withUriSegment(entityTypeId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a new permission for an entity type. You must specify the Id of the entity type you are creating the permission for.
     * You can optionally specify an Id for the permission inside the EntityTypePermission object itself, if not provided one will be generated.
     *
     * @param {UUID} entityTypeId The Id of the entity type to create the permission on.
     * @param {UUID} permissionId (Optional) The Id of the permission. If not provided a secure random UUID will be generated.
     * @param {EntityTypeRequest} request The request object that contains all the information used to create the permission.
     * @returns {Promise<ClientResponse<EntityTypeResponse>>}
     */
    createEntityTypePermission(entityTypeId, permissionId, request) {
        return this.start()
            .withUri('/api/entity/type')
            .withUriSegment(entityTypeId)
            .withUriSegment("permission")
            .withUriSegment(permissionId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a family with the user Id in the request as the owner and sole member of the family. You can optionally specify an Id for the
     * family, if not provided one will be generated.
     *
     * @param {UUID} familyId (Optional) The Id for the family. If not provided a secure random UUID will be generated.
     * @param {FamilyRequest} request The request object that contains all the information used to create the family.
     * @returns {Promise<ClientResponse<FamilyResponse>>}
     */
    createFamily(familyId, request) {
        return this.start()
            .withUri('/api/user/family')
            .withUriSegment(familyId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a form.  You can optionally specify an Id for the form, if not provided one will be generated.
     *
     * @param {UUID} formId (Optional) The Id for the form. If not provided a secure random UUID will be generated.
     * @param {FormRequest} request The request object that contains all the information used to create the form.
     * @returns {Promise<ClientResponse<FormResponse>>}
     */
    createForm(formId, request) {
        return this.start()
            .withUri('/api/form')
            .withUriSegment(formId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a form field.  You can optionally specify an Id for the form, if not provided one will be generated.
     *
     * @param {UUID} fieldId (Optional) The Id for the form field. If not provided a secure random UUID will be generated.
     * @param {FormFieldRequest} request The request object that contains all the information used to create the form field.
     * @returns {Promise<ClientResponse<FormFieldResponse>>}
     */
    createFormField(fieldId, request) {
        return this.start()
            .withUri('/api/form/field')
            .withUriSegment(fieldId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a group. You can optionally specify an Id for the group, if not provided one will be generated.
     *
     * @param {UUID} groupId (Optional) The Id for the group. If not provided a secure random UUID will be generated.
     * @param {GroupRequest} request The request object that contains all the information used to create the group.
     * @returns {Promise<ClientResponse<GroupResponse>>}
     */
    createGroup(groupId, request) {
        return this.start()
            .withUri('/api/group')
            .withUriSegment(groupId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a member in a group.
     *
     * @param {MemberRequest} request The request object that contains all the information used to create the group member(s).
     * @returns {Promise<ClientResponse<MemberResponse>>}
     */
    createGroupMembers(request) {
        return this.start()
            .withUri('/api/group/member')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates an IP Access Control List. You can optionally specify an Id on this create request, if one is not provided one will be generated.
     *
     * @param {UUID} accessControlListId (Optional) The Id for the IP Access Control List. If not provided a secure random UUID will be generated.
     * @param {IPAccessControlListRequest} request The request object that contains all the information used to create the IP Access Control List.
     * @returns {Promise<ClientResponse<IPAccessControlListResponse>>}
     */
    createIPAccessControlList(accessControlListId, request) {
        return this.start()
            .withUri('/api/ip-acl')
            .withUriSegment(accessControlListId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates an identity provider. You can optionally specify an Id for the identity provider, if not provided one will be generated.
     *
     * @param {UUID} identityProviderId (Optional) The Id of the identity provider. If not provided a secure random UUID will be generated.
     * @param {IdentityProviderRequest} request The request object that contains all the information used to create the identity provider.
     * @returns {Promise<ClientResponse<IdentityProviderResponse>>}
     */
    createIdentityProvider(identityProviderId, request) {
        return this.start()
            .withUri('/api/identity-provider')
            .withUriSegment(identityProviderId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a Lambda. You can optionally specify an Id for the lambda, if not provided one will be generated.
     *
     * @param {UUID} lambdaId (Optional) The Id for the lambda. If not provided a secure random UUID will be generated.
     * @param {LambdaRequest} request The request object that contains all the information used to create the lambda.
     * @returns {Promise<ClientResponse<LambdaResponse>>}
     */
    createLambda(lambdaId, request) {
        return this.start()
            .withUri('/api/lambda')
            .withUriSegment(lambdaId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates an message template. You can optionally specify an Id for the template, if not provided one will be generated.
     *
     * @param {UUID} messageTemplateId (Optional) The Id for the template. If not provided a secure random UUID will be generated.
     * @param {MessageTemplateRequest} request The request object that contains all the information used to create the message template.
     * @returns {Promise<ClientResponse<MessageTemplateResponse>>}
     */
    createMessageTemplate(messageTemplateId, request) {
        return this.start()
            .withUri('/api/message/template')
            .withUriSegment(messageTemplateId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a messenger.  You can optionally specify an Id for the messenger, if not provided one will be generated.
     *
     * @param {UUID} messengerId (Optional) The Id for the messenger. If not provided a secure random UUID will be generated.
     * @param {MessengerRequest} request The request object that contains all the information used to create the messenger.
     * @returns {Promise<ClientResponse<MessengerResponse>>}
     */
    createMessenger(messengerId, request) {
        return this.start()
            .withUri('/api/messenger')
            .withUriSegment(messengerId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a new custom OAuth scope for an application. You must specify the Id of the application you are creating the scope for.
     * You can optionally specify an Id for the OAuth scope on the URL, if not provided one will be generated.
     *
     * @param {UUID} applicationId The Id of the application to create the OAuth scope on.
     * @param {UUID} scopeId (Optional) The Id of the OAuth scope. If not provided a secure random UUID will be generated.
     * @param {ApplicationOAuthScopeRequest} request The request object that contains all the information used to create the OAuth OAuth scope.
     * @returns {Promise<ClientResponse<ApplicationOAuthScopeResponse>>}
     */
    createOAuthScope(applicationId, scopeId, request) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withUriSegment("scope")
            .withUriSegment(scopeId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a tenant. You can optionally specify an Id for the tenant, if not provided one will be generated.
     *
     * @param {UUID} tenantId (Optional) The Id for the tenant. If not provided a secure random UUID will be generated.
     * @param {TenantRequest} request The request object that contains all the information used to create the tenant.
     * @returns {Promise<ClientResponse<TenantResponse>>}
     */
    createTenant(tenantId, request) {
        return this.start()
            .withUri('/api/tenant')
            .withUriSegment(tenantId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a Theme. You can optionally specify an Id for the theme, if not provided one will be generated.
     *
     * @param {UUID} themeId (Optional) The Id for the theme. If not provided a secure random UUID will be generated.
     * @param {ThemeRequest} request The request object that contains all the information used to create the theme.
     * @returns {Promise<ClientResponse<ThemeResponse>>}
     */
    createTheme(themeId, request) {
        return this.start()
            .withUri('/api/theme')
            .withUriSegment(themeId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a user. You can optionally specify an Id for the user, if not provided one will be generated.
     *
     * @param {UUID} userId (Optional) The Id for the user. If not provided a secure random UUID will be generated.
     * @param {UserRequest} request The request object that contains all the information used to create the user.
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    createUser(userId, request) {
        return this.start()
            .withUri('/api/user')
            .withUriSegment(userId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a user action. This action cannot be taken on a user until this call successfully returns. Anytime after
     * that the user action can be applied to any user.
     *
     * @param {UUID} userActionId (Optional) The Id for the user action. If not provided a secure random UUID will be generated.
     * @param {UserActionRequest} request The request object that contains all the information used to create the user action.
     * @returns {Promise<ClientResponse<UserActionResponse>>}
     */
    createUserAction(userActionId, request) {
        return this.start()
            .withUri('/api/user-action')
            .withUriSegment(userActionId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a user reason. This user action reason cannot be used when actioning a user until this call completes
     * successfully. Anytime after that the user action reason can be used.
     *
     * @param {UUID} userActionReasonId (Optional) The Id for the user action reason. If not provided a secure random UUID will be generated.
     * @param {UserActionReasonRequest} request The request object that contains all the information used to create the user action reason.
     * @returns {Promise<ClientResponse<UserActionReasonResponse>>}
     */
    createUserActionReason(userActionReasonId, request) {
        return this.start()
            .withUri('/api/user-action-reason')
            .withUriSegment(userActionReasonId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a single User consent.
     *
     * @param {UUID} userConsentId (Optional) The Id for the User consent. If not provided a secure random UUID will be generated.
     * @param {UserConsentRequest} request The request that contains the user consent information.
     * @returns {Promise<ClientResponse<UserConsentResponse>>}
     */
    createUserConsent(userConsentId, request) {
        return this.start()
            .withUri('/api/user/consent')
            .withUriSegment(userConsentId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Link an external user from a 3rd party identity provider to a FusionAuth user.
     *
     * @param {IdentityProviderLinkRequest} request The request object that contains all the information used to link the FusionAuth user.
     * @returns {Promise<ClientResponse<IdentityProviderLinkResponse>>}
     */
    createUserLink(request) {
        return this.start()
            .withUri('/api/identity-provider/link')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Creates a webhook. You can optionally specify an Id for the webhook, if not provided one will be generated.
     *
     * @param {UUID} webhookId (Optional) The Id for the webhook. If not provided a secure random UUID will be generated.
     * @param {WebhookRequest} request The request object that contains all the information used to create the webhook.
     * @returns {Promise<ClientResponse<WebhookResponse>>}
     */
    createWebhook(webhookId, request) {
        return this.start()
            .withUri('/api/webhook')
            .withUriSegment(webhookId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Deactivates the application with the given Id.
     *
     * @param {UUID} applicationId The Id of the application to deactivate.
     * @returns {Promise<ClientResponse<void>>}
     */
    deactivateApplication(applicationId) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deactivates the FusionAuth Reactor.
     *
     * @returns {Promise<ClientResponse<void>>}
     */
    deactivateReactor() {
        return this.start()
            .withUri('/api/reactor')
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deactivates the user with the given Id.
     *
     * @param {UUID} userId The Id of the user to deactivate.
     * @returns {Promise<ClientResponse<void>>}
     */
    deactivateUser(userId) {
        return this.start()
            .withUri('/api/user')
            .withUriSegment(userId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deactivates the user action with the given Id.
     *
     * @param {UUID} userActionId The Id of the user action to deactivate.
     * @returns {Promise<ClientResponse<void>>}
     */
    deactivateUserAction(userActionId) {
        return this.start()
            .withUri('/api/user-action')
            .withUriSegment(userActionId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deactivates the users with the given Ids.
     *
     * @param {Array<string>} userIds The ids of the users to deactivate.
     * @returns {Promise<ClientResponse<UserDeleteResponse>>}
     *
     * @deprecated This method has been renamed to deactivateUsersByIds, use that method instead.
     */
    deactivateUsers(userIds) {
        return this.start()
            .withUri('/api/user/bulk')
            .withParameter('userId', userIds)
            .withParameter('dryRun', false)
            .withParameter('hardDelete', false)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deactivates the users with the given Ids.
     *
     * @param {Array<string>} userIds The ids of the users to deactivate.
     * @returns {Promise<ClientResponse<UserDeleteResponse>>}
     */
    deactivateUsersByIds(userIds) {
        return this.start()
            .withUri('/api/user/bulk')
            .withParameter('userId', userIds)
            .withParameter('dryRun', false)
            .withParameter('hardDelete', false)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the API key for the given Id.
     *
     * @param {UUID} keyId The Id of the authentication API key to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteAPIKey(keyId) {
        return this.start()
            .withUri('/api/api-key')
            .withUriSegment(keyId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Hard deletes an application. This is a dangerous operation and should not be used in most circumstances. This will
     * delete the application, any registrations for that application, metrics and reports for the application, all the
     * roles for the application, and any other data associated with the application. This operation could take a very
     * long time, depending on the amount of data in your database.
     *
     * @param {UUID} applicationId The Id of the application to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteApplication(applicationId) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withParameter('hardDelete', true)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Hard deletes an application role. This is a dangerous operation and should not be used in most circumstances. This
     * permanently removes the given role from all users that had it.
     *
     * @param {UUID} applicationId The Id of the application that the role belongs to.
     * @param {UUID} roleId The Id of the role to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteApplicationRole(applicationId, roleId) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withUriSegment("role")
            .withUriSegment(roleId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the connector for the given Id.
     *
     * @param {UUID} connectorId The Id of the connector to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteConnector(connectorId) {
        return this.start()
            .withUri('/api/connector')
            .withUriSegment(connectorId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the consent for the given Id.
     *
     * @param {UUID} consentId The Id of the consent to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteConsent(consentId) {
        return this.start()
            .withUri('/api/consent')
            .withUriSegment(consentId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the email template for the given Id.
     *
     * @param {UUID} emailTemplateId The Id of the email template to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteEmailTemplate(emailTemplateId) {
        return this.start()
            .withUri('/api/email/template')
            .withUriSegment(emailTemplateId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the Entity for the given Id.
     *
     * @param {UUID} entityId The Id of the Entity to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteEntity(entityId) {
        return this.start()
            .withUri('/api/entity')
            .withUriSegment(entityId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes an Entity Grant for the given User or Entity.
     *
     * @param {UUID} entityId The Id of the Entity that the Entity Grant is being deleted for.
     * @param {UUID} recipientEntityId (Optional) The Id of the Entity that the Entity Grant is for.
     * @param {UUID} userId (Optional) The Id of the User that the Entity Grant is for.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteEntityGrant(entityId, recipientEntityId, userId) {
        return this.start()
            .withUri('/api/entity')
            .withUriSegment(entityId)
            .withUriSegment("grant")
            .withParameter('recipientEntityId', recipientEntityId)
            .withParameter('userId', userId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the Entity Type for the given Id.
     *
     * @param {UUID} entityTypeId The Id of the Entity Type to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteEntityType(entityTypeId) {
        return this.start()
            .withUri('/api/entity/type')
            .withUriSegment(entityTypeId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Hard deletes a permission. This is a dangerous operation and should not be used in most circumstances. This
     * permanently removes the given permission from all grants that had it.
     *
     * @param {UUID} entityTypeId The Id of the entityType the the permission belongs to.
     * @param {UUID} permissionId The Id of the permission to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteEntityTypePermission(entityTypeId, permissionId) {
        return this.start()
            .withUri('/api/entity/type')
            .withUriSegment(entityTypeId)
            .withUriSegment("permission")
            .withUriSegment(permissionId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the form for the given Id.
     *
     * @param {UUID} formId The Id of the form to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteForm(formId) {
        return this.start()
            .withUri('/api/form')
            .withUriSegment(formId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the form field for the given Id.
     *
     * @param {UUID} fieldId The Id of the form field to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteFormField(fieldId) {
        return this.start()
            .withUri('/api/form/field')
            .withUriSegment(fieldId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the group for the given Id.
     *
     * @param {UUID} groupId The Id of the group to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteGroup(groupId) {
        return this.start()
            .withUri('/api/group')
            .withUriSegment(groupId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Removes users as members of a group.
     *
     * @param {MemberDeleteRequest} request The member request that contains all the information used to remove members to the group.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteGroupMembers(request) {
        return this.start()
            .withUri('/api/group/member')
            .withJSONBody(request)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the IP Access Control List for the given Id.
     *
     * @param {UUID} ipAccessControlListId The Id of the IP Access Control List to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteIPAccessControlList(ipAccessControlListId) {
        return this.start()
            .withUri('/api/ip-acl')
            .withUriSegment(ipAccessControlListId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the identity provider for the given Id.
     *
     * @param {UUID} identityProviderId The Id of the identity provider to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteIdentityProvider(identityProviderId) {
        return this.start()
            .withUri('/api/identity-provider')
            .withUriSegment(identityProviderId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the key for the given Id.
     *
     * @param {UUID} keyId The Id of the key to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteKey(keyId) {
        return this.start()
            .withUri('/api/key')
            .withUriSegment(keyId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the lambda for the given Id.
     *
     * @param {UUID} lambdaId The Id of the lambda to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteLambda(lambdaId) {
        return this.start()
            .withUri('/api/lambda')
            .withUriSegment(lambdaId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the message template for the given Id.
     *
     * @param {UUID} messageTemplateId The Id of the message template to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteMessageTemplate(messageTemplateId) {
        return this.start()
            .withUri('/api/message/template')
            .withUriSegment(messageTemplateId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the messenger for the given Id.
     *
     * @param {UUID} messengerId The Id of the messenger to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteMessenger(messengerId) {
        return this.start()
            .withUri('/api/messenger')
            .withUriSegment(messengerId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Hard deletes a custom OAuth scope.
     * OAuth workflows that are still requesting the deleted OAuth scope may fail depending on the application's unknown scope policy.
     *
     * @param {UUID} applicationId The Id of the application that the OAuth scope belongs to.
     * @param {UUID} scopeId The Id of the OAuth scope to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteOAuthScope(applicationId, scopeId) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withUriSegment("scope")
            .withUriSegment(scopeId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the user registration for the given user and application.
     *
     * @param {UUID} userId The Id of the user whose registration is being deleted.
     * @param {UUID} applicationId The Id of the application to remove the registration for.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteRegistration(userId, applicationId) {
        return this.start()
            .withUri('/api/user/registration')
            .withUriSegment(userId)
            .withUriSegment(applicationId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the user registration for the given user and application along with the given JSON body that contains the event information.
     *
     * @param {UUID} userId The Id of the user whose registration is being deleted.
     * @param {UUID} applicationId The Id of the application to remove the registration for.
     * @param {RegistrationDeleteRequest} request The request body that contains the event information.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteRegistrationWithRequest(userId, applicationId, request) {
        return this.start()
            .withUri('/api/user/registration')
            .withUriSegment(userId)
            .withUriSegment(applicationId)
            .withJSONBody(request)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the tenant based on the given Id on the URL. This permanently deletes all information, metrics, reports and data associated
     * with the tenant and everything under the tenant (applications, users, etc).
     *
     * @param {UUID} tenantId The Id of the tenant to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteTenant(tenantId) {
        return this.start()
            .withUri('/api/tenant')
            .withUriSegment(tenantId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the tenant for the given Id asynchronously.
     * This method is helpful if you do not want to wait for the delete operation to complete.
     *
     * @param {UUID} tenantId The Id of the tenant to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteTenantAsync(tenantId) {
        return this.start()
            .withUri('/api/tenant')
            .withUriSegment(tenantId)
            .withParameter('async', true)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the tenant based on the given request (sent to the API as JSON). This permanently deletes all information, metrics, reports and data associated
     * with the tenant and everything under the tenant (applications, users, etc).
     *
     * @param {UUID} tenantId The Id of the tenant to delete.
     * @param {TenantDeleteRequest} request The request object that contains all the information used to delete the user.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteTenantWithRequest(tenantId, request) {
        return this.start()
            .withUri('/api/tenant')
            .withUriSegment(tenantId)
            .withJSONBody(request)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the theme for the given Id.
     *
     * @param {UUID} themeId The Id of the theme to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteTheme(themeId) {
        return this.start()
            .withUri('/api/theme')
            .withUriSegment(themeId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the user for the given Id. This permanently deletes all information, metrics, reports and data associated
     * with the user.
     *
     * @param {UUID} userId The Id of the user to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteUser(userId) {
        return this.start()
            .withUri('/api/user')
            .withUriSegment(userId)
            .withParameter('hardDelete', true)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the user action for the given Id. This permanently deletes the user action and also any history and logs of
     * the action being applied to any users.
     *
     * @param {UUID} userActionId The Id of the user action to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteUserAction(userActionId) {
        return this.start()
            .withUri('/api/user-action')
            .withUriSegment(userActionId)
            .withParameter('hardDelete', true)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the user action reason for the given Id.
     *
     * @param {UUID} userActionReasonId The Id of the user action reason to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteUserActionReason(userActionReasonId) {
        return this.start()
            .withUri('/api/user-action-reason')
            .withUriSegment(userActionReasonId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Remove an existing link that has been made from a 3rd party identity provider to a FusionAuth user.
     *
     * @param {UUID} identityProviderId The unique Id of the identity provider.
     * @param {string} identityProviderUserId The unique Id of the user in the 3rd party identity provider to unlink.
     * @param {UUID} userId The unique Id of the FusionAuth user to unlink.
     * @returns {Promise<ClientResponse<IdentityProviderLinkResponse>>}
     */
    deleteUserLink(identityProviderId, identityProviderUserId, userId) {
        return this.start()
            .withUri('/api/identity-provider/link')
            .withParameter('identityProviderId', identityProviderId)
            .withParameter('identityProviderUserId', identityProviderUserId)
            .withParameter('userId', userId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the user based on the given request (sent to the API as JSON). This permanently deletes all information, metrics, reports and data associated
     * with the user.
     *
     * @param {UUID} userId The Id of the user to delete (required).
     * @param {UserDeleteSingleRequest} request The request object that contains all the information used to delete the user.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteUserWithRequest(userId, request) {
        return this.start()
            .withUri('/api/user')
            .withUriSegment(userId)
            .withJSONBody(request)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the users with the given Ids, or users matching the provided JSON query or queryString.
     * The order of preference is Ids, query and then queryString, it is recommended to only provide one of the three for the request.
     *
     * This method can be used to deactivate or permanently delete (hard-delete) users based upon the hardDelete boolean in the request body.
     * Using the dryRun parameter you may also request the result of the action without actually deleting or deactivating any users.
     *
     * @param {UserDeleteRequest} request The UserDeleteRequest.
     * @returns {Promise<ClientResponse<UserDeleteResponse>>}
     *
     * @deprecated This method has been renamed to deleteUsersByQuery, use that method instead.
     */
    deleteUsers(request) {
        return this.start()
            .withUri('/api/user/bulk')
            .withJSONBody(request)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the users with the given Ids, or users matching the provided JSON query or queryString.
     * The order of preference is Ids, query and then queryString, it is recommended to only provide one of the three for the request.
     *
     * This method can be used to deactivate or permanently delete (hard-delete) users based upon the hardDelete boolean in the request body.
     * Using the dryRun parameter you may also request the result of the action without actually deleting or deactivating any users.
     *
     * @param {UserDeleteRequest} request The UserDeleteRequest.
     * @returns {Promise<ClientResponse<UserDeleteResponse>>}
     */
    deleteUsersByQuery(request) {
        return this.start()
            .withUri('/api/user/bulk')
            .withJSONBody(request)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the WebAuthn credential for the given Id.
     *
     * @param {UUID} id The Id of the WebAuthn credential to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteWebAuthnCredential(id) {
        return this.start()
            .withUri('/api/webauthn')
            .withUriSegment(id)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Deletes the webhook for the given Id.
     *
     * @param {UUID} webhookId The Id of the webhook to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    deleteWebhook(webhookId) {
        return this.start()
            .withUri('/api/webhook')
            .withUriSegment(webhookId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Disable two-factor authentication for a user.
     *
     * @param {UUID} userId The Id of the User for which you're disabling two-factor authentication.
     * @param {string} methodId The two-factor method identifier you wish to disable
     * @param {string} code The two-factor code used verify the the caller knows the two-factor secret.
     * @returns {Promise<ClientResponse<void>>}
     */
    disableTwoFactor(userId, methodId, code) {
        return this.start()
            .withUri('/api/user/two-factor')
            .withUriSegment(userId)
            .withParameter('methodId', methodId)
            .withParameter('code', code)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Disable two-factor authentication for a user using a JSON body rather than URL parameters.
     *
     * @param {UUID} userId The Id of the User for which you're disabling two-factor authentication.
     * @param {TwoFactorDisableRequest} request The request information that contains the code and methodId along with any event information.
     * @returns {Promise<ClientResponse<void>>}
     */
    disableTwoFactorWithRequest(userId, request) {
        return this.start()
            .withUri('/api/user/two-factor')
            .withUriSegment(userId)
            .withJSONBody(request)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Enable two-factor authentication for a user.
     *
     * @param {UUID} userId The Id of the user to enable two-factor authentication.
     * @param {TwoFactorRequest} request The two-factor enable request information.
     * @returns {Promise<ClientResponse<TwoFactorResponse>>}
     */
    enableTwoFactor(userId, request) {
        return this.start()
            .withUri('/api/user/two-factor')
            .withUriSegment(userId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Exchanges an OAuth authorization code for an access token.
     * Makes a request to the Token endpoint to exchange the authorization code returned from the Authorize endpoint for an access token.
     *
     * @param {string} code The authorization code returned on the /oauth2/authorize response.
     * @param {string} client_id (Optional) The unique client identifier. The client Id is the Id of the FusionAuth Application in which you are attempting to authenticate.
     *    This parameter is optional when Basic Authorization is used to authenticate this request.
     * @param {string} client_secret (Optional) The client secret. This value will be required if client authentication is enabled.
     * @param {string} redirect_uri The URI to redirect to upon a successful request.
     * @returns {Promise<ClientResponse<AccessToken>>}
     */
    exchangeOAuthCodeForAccessToken(code, client_id, client_secret, redirect_uri) {
        let body = new url_1.URLSearchParams();
        body.append('code', code);
        body.append('client_id', client_id);
        body.append('client_secret', client_secret);
        body.append('grant_type', 'authorization_code');
        body.append('redirect_uri', redirect_uri);
        return this.startAnonymous()
            .withUri('/oauth2/token')
            .withFormData(body)
            .withMethod("POST")
            .go();
    }
    /**
     * Exchanges an OAuth authorization code and code_verifier for an access token.
     * Makes a request to the Token endpoint to exchange the authorization code returned from the Authorize endpoint and a code_verifier for an access token.
     *
     * @param {string} code The authorization code returned on the /oauth2/authorize response.
     * @param {string} client_id (Optional) The unique client identifier. The client Id is the Id of the FusionAuth Application in which you are attempting to authenticate. This parameter is optional when the Authorization header is provided.
     *    This parameter is optional when Basic Authorization is used to authenticate this request.
     * @param {string} client_secret (Optional) The client secret. This value may optionally be provided in the request body instead of the Authorization header.
     * @param {string} redirect_uri The URI to redirect to upon a successful request.
     * @param {string} code_verifier The random string generated previously. Will be compared with the code_challenge sent previously, which allows the OAuth provider to authenticate your app.
     * @returns {Promise<ClientResponse<AccessToken>>}
     */
    exchangeOAuthCodeForAccessTokenUsingPKCE(code, client_id, client_secret, redirect_uri, code_verifier) {
        let body = new url_1.URLSearchParams();
        body.append('code', code);
        body.append('client_id', client_id);
        body.append('client_secret', client_secret);
        body.append('grant_type', 'authorization_code');
        body.append('redirect_uri', redirect_uri);
        body.append('code_verifier', code_verifier);
        return this.startAnonymous()
            .withUri('/oauth2/token')
            .withFormData(body)
            .withMethod("POST")
            .go();
    }
    /**
     * Exchange a Refresh Token for an Access Token.
     * If you will be using the Refresh Token Grant, you will make a request to the Token endpoint to exchange the user’s refresh token for an access token.
     *
     * @param {string} refresh_token The refresh token that you would like to use to exchange for an access token.
     * @param {string} client_id (Optional) The unique client identifier. The client Id is the Id of the FusionAuth Application in which you are attempting to authenticate. This parameter is optional when the Authorization header is provided.
     *    This parameter is optional when Basic Authorization is used to authenticate this request.
     * @param {string} client_secret (Optional) The client secret. This value may optionally be provided in the request body instead of the Authorization header.
     * @param {string} scope (Optional) This parameter is optional and if omitted, the same scope requested during the authorization request will be used. If provided the scopes must match those requested during the initial authorization request.
     * @param {string} user_code (Optional) The end-user verification code. This code is required if using this endpoint to approve the Device Authorization.
     * @returns {Promise<ClientResponse<AccessToken>>}
     */
    exchangeRefreshTokenForAccessToken(refresh_token, client_id, client_secret, scope, user_code) {
        let body = new url_1.URLSearchParams();
        body.append('refresh_token', refresh_token);
        body.append('client_id', client_id);
        body.append('client_secret', client_secret);
        body.append('grant_type', 'refresh_token');
        body.append('scope', scope);
        body.append('user_code', user_code);
        return this.startAnonymous()
            .withUri('/oauth2/token')
            .withFormData(body)
            .withMethod("POST")
            .go();
    }
    /**
     * Exchange a refresh token for a new JWT.
     *
     * @param {RefreshRequest} request The refresh request.
     * @returns {Promise<ClientResponse<JWTRefreshResponse>>}
     */
    exchangeRefreshTokenForJWT(request) {
        return this.startAnonymous()
            .withUri('/api/jwt/refresh')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Exchange User Credentials for a Token.
     * If you will be using the Resource Owner Password Credential Grant, you will make a request to the Token endpoint to exchange the user’s email and password for an access token.
     *
     * @param {string} username The login identifier of the user. The login identifier can be either the email or the username.
     * @param {string} password The user’s password.
     * @param {string} client_id (Optional) The unique client identifier. The client Id is the Id of the FusionAuth Application in which you are attempting to authenticate. This parameter is optional when the Authorization header is provided.
     *    This parameter is optional when Basic Authorization is used to authenticate this request.
     * @param {string} client_secret (Optional) The client secret. This value may optionally be provided in the request body instead of the Authorization header.
     * @param {string} scope (Optional) This parameter is optional and if omitted, the same scope requested during the authorization request will be used. If provided the scopes must match those requested during the initial authorization request.
     * @param {string} user_code (Optional) The end-user verification code. This code is required if using this endpoint to approve the Device Authorization.
     * @returns {Promise<ClientResponse<AccessToken>>}
     */
    exchangeUserCredentialsForAccessToken(username, password, client_id, client_secret, scope, user_code) {
        let body = new url_1.URLSearchParams();
        body.append('username', username);
        body.append('password', password);
        body.append('client_id', client_id);
        body.append('client_secret', client_secret);
        body.append('grant_type', 'password');
        body.append('scope', scope);
        body.append('user_code', user_code);
        return this.startAnonymous()
            .withUri('/oauth2/token')
            .withFormData(body)
            .withMethod("POST")
            .go();
    }
    /**
     * Begins the forgot password sequence, which kicks off an email to the user so that they can reset their password.
     *
     * @param {ForgotPasswordRequest} request The request that contains the information about the user so that they can be emailed.
     * @returns {Promise<ClientResponse<ForgotPasswordResponse>>}
     */
    forgotPassword(request) {
        return this.start()
            .withUri('/api/user/forgot-password')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Generate a new Email Verification Id to be used with the Verify Email API. This API will not attempt to send an
     * email to the User. This API may be used to collect the verificationId for use with a third party system.
     *
     * @param {string} email The email address of the user that needs a new verification email.
     * @returns {Promise<ClientResponse<VerifyEmailResponse>>}
     */
    generateEmailVerificationId(email) {
        return this.start()
            .withUri('/api/user/verify-email')
            .withParameter('email', email)
            .withParameter('sendVerifyEmail', false)
            .withMethod("PUT")
            .go();
    }
    /**
     * Generate a new RSA or EC key pair or an HMAC secret.
     *
     * @param {UUID} keyId (Optional) The Id for the key. If not provided a secure random UUID will be generated.
     * @param {KeyRequest} request The request object that contains all the information used to create the key.
     * @returns {Promise<ClientResponse<KeyResponse>>}
     */
    generateKey(keyId, request) {
        return this.start()
            .withUri('/api/key/generate')
            .withUriSegment(keyId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Generate a new Application Registration Verification Id to be used with the Verify Registration API. This API will not attempt to send an
     * email to the User. This API may be used to collect the verificationId for use with a third party system.
     *
     * @param {string} email The email address of the user that needs a new verification email.
     * @param {UUID} applicationId The Id of the application to be verified.
     * @returns {Promise<ClientResponse<VerifyRegistrationResponse>>}
     */
    generateRegistrationVerificationId(email, applicationId) {
        return this.start()
            .withUri('/api/user/verify-registration')
            .withParameter('email', email)
            .withParameter('sendVerifyPasswordEmail', false)
            .withParameter('applicationId', applicationId)
            .withMethod("PUT")
            .go();
    }
    /**
     * Generate two-factor recovery codes for a user. Generating two-factor recovery codes will invalidate any existing recovery codes.
     *
     * @param {UUID} userId The Id of the user to generate new Two Factor recovery codes.
     * @returns {Promise<ClientResponse<TwoFactorRecoveryCodeResponse>>}
     */
    generateTwoFactorRecoveryCodes(userId) {
        return this.start()
            .withHeader('Content-Type', 'text/plain')
            .withUri('/api/user/two-factor/recovery-code')
            .withUriSegment(userId)
            .withMethod("POST")
            .go();
    }
    /**
     * Generate a Two Factor secret that can be used to enable Two Factor authentication for a User. The response will contain
     * both the secret and a Base32 encoded form of the secret which can be shown to a User when using a 2 Step Authentication
     * application such as Google Authenticator.
     *
     * @returns {Promise<ClientResponse<SecretResponse>>}
     */
    generateTwoFactorSecret() {
        return this.start()
            .withUri('/api/two-factor/secret')
            .withMethod("GET")
            .go();
    }
    /**
     * Generate a Two Factor secret that can be used to enable Two Factor authentication for a User. The response will contain
     * both the secret and a Base32 encoded form of the secret which can be shown to a User when using a 2 Step Authentication
     * application such as Google Authenticator.
     *
     * @param {string} encodedJWT The encoded JWT (access token).
     * @returns {Promise<ClientResponse<SecretResponse>>}
     */
    generateTwoFactorSecretUsingJWT(encodedJWT) {
        return this.startAnonymous()
            .withUri('/api/two-factor/secret')
            .withAuthorization('Bearer ' + encodedJWT)
            .withMethod("GET")
            .go();
    }
    /**
     * Handles login via third-parties including Social login, external OAuth and OpenID Connect, and other
     * login systems.
     *
     * @param {IdentityProviderLoginRequest} request The third-party login request that contains information from the third-party login
     *    providers that FusionAuth uses to reconcile the user's account.
     * @returns {Promise<ClientResponse<LoginResponse>>}
     */
    identityProviderLogin(request) {
        return this.startAnonymous()
            .withUri('/api/identity-provider/login')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Import an existing RSA or EC key pair or an HMAC secret.
     *
     * @param {UUID} keyId (Optional) The Id for the key. If not provided a secure random UUID will be generated.
     * @param {KeyRequest} request The request object that contains all the information used to create the key.
     * @returns {Promise<ClientResponse<KeyResponse>>}
     */
    importKey(keyId, request) {
        return this.start()
            .withUri('/api/key/import')
            .withUriSegment(keyId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Bulk imports refresh tokens. This request performs minimal validation and runs batch inserts of refresh tokens with the
     * expectation that each token represents a user that already exists and is registered for the corresponding FusionAuth
     * Application. This is done to increases the insert performance.
     *
     * Therefore, if you encounter an error due to a database key violation, the response will likely offer a generic
     * explanation. If you encounter an error, you may optionally enable additional validation to receive a JSON response
     * body with specific validation errors. This will slow the request down but will allow you to identify the cause of
     * the failure. See the validateDbConstraints request parameter.
     *
     * @param {RefreshTokenImportRequest} request The request that contains all the information about all the refresh tokens to import.
     * @returns {Promise<ClientResponse<void>>}
     */
    importRefreshTokens(request) {
        return this.start()
            .withUri('/api/user/refresh-token/import')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Bulk imports users. This request performs minimal validation and runs batch inserts of users with the expectation
     * that each user does not yet exist and each registration corresponds to an existing FusionAuth Application. This is done to
     * increases the insert performance.
     *
     * Therefore, if you encounter an error due to a database key violation, the response will likely offer
     * a generic explanation. If you encounter an error, you may optionally enable additional validation to receive a JSON response
     * body with specific validation errors. This will slow the request down but will allow you to identify the cause of the failure. See
     * the validateDbConstraints request parameter.
     *
     * @param {ImportRequest} request The request that contains all the information about all the users to import.
     * @returns {Promise<ClientResponse<void>>}
     */
    importUsers(request) {
        return this.start()
            .withUri('/api/user/import')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Import a WebAuthn credential
     *
     * @param {WebAuthnCredentialImportRequest} request An object containing data necessary for importing the credential
     * @returns {Promise<ClientResponse<void>>}
     */
    importWebAuthnCredential(request) {
        return this.start()
            .withUri('/api/webauthn/import')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Inspect an access token issued as the result of the User based grant such as the Authorization Code Grant, Implicit Grant, the User Credentials Grant or the Refresh Grant.
     *
     * @param {string} client_id The unique client identifier. The client Id is the Id of the FusionAuth Application for which this token was generated.
     * @param {string} token The access token returned by this OAuth provider as the result of a successful client credentials grant.
     * @returns {Promise<ClientResponse<IntrospectResponse>>}
     */
    introspectAccessToken(client_id, token) {
        let body = new url_1.URLSearchParams();
        body.append('client_id', client_id);
        body.append('token', token);
        return this.startAnonymous()
            .withUri('/oauth2/introspect')
            .withFormData(body)
            .withMethod("POST")
            .go();
    }
    /**
     * Inspect an access token issued as the result of the Client Credentials Grant.
     *
     * @param {string} token The access token returned by this OAuth provider as the result of a successful client credentials grant.
     * @returns {Promise<ClientResponse<IntrospectResponse>>}
     */
    introspectClientCredentialsAccessToken(token) {
        let body = new url_1.URLSearchParams();
        body.append('token', token);
        return this.startAnonymous()
            .withUri('/oauth2/introspect')
            .withFormData(body)
            .withMethod("POST")
            .go();
    }
    /**
     * Issue a new access token (JWT) for the requested Application after ensuring the provided JWT is valid. A valid
     * access token is properly signed and not expired.
     * <p>
     * This API may be used in an SSO configuration to issue new tokens for another application after the user has
     * obtained a valid token from authentication.
     *
     * @param {UUID} applicationId The Application Id for which you are requesting a new access token be issued.
     * @param {string} encodedJWT The encoded JWT (access token).
     * @param {string} refreshToken (Optional) An existing refresh token used to request a refresh token in addition to a JWT in the response.
     *    <p>The target application represented by the applicationId request parameter must have refresh
     *    tokens enabled in order to receive a refresh token in the response.</p>
     * @returns {Promise<ClientResponse<IssueResponse>>}
     */
    issueJWT(applicationId, encodedJWT, refreshToken) {
        return this.startAnonymous()
            .withUri('/api/jwt/issue')
            .withAuthorization('Bearer ' + encodedJWT)
            .withParameter('applicationId', applicationId)
            .withParameter('refreshToken', refreshToken)
            .withMethod("GET")
            .go();
    }
    /**
     * Authenticates a user to FusionAuth.
     *
     * This API optionally requires an API key. See <code>Application.loginConfiguration.requireAuthentication</code>.
     *
     * @param {LoginRequest} request The login request that contains the user credentials used to log them in.
     * @returns {Promise<ClientResponse<LoginResponse>>}
     */
    login(request) {
        return this.start()
            .withUri('/api/login')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Sends a ping to FusionAuth indicating that the user was automatically logged into an application. When using
     * FusionAuth's SSO or your own, you should call this if the user is already logged in centrally, but accesses an
     * application where they no longer have a session. This helps correctly track login counts, times and helps with
     * reporting.
     *
     * @param {UUID} userId The Id of the user that was logged in.
     * @param {UUID} applicationId The Id of the application that they logged into.
     * @param {string} callerIPAddress (Optional) The IP address of the end-user that is logging in. If a null value is provided
     *    the IP address will be that of the client or last proxy that sent the request.
     * @returns {Promise<ClientResponse<LoginResponse>>}
     */
    loginPing(userId, applicationId, callerIPAddress) {
        return this.start()
            .withUri('/api/login')
            .withUriSegment(userId)
            .withUriSegment(applicationId)
            .withParameter('ipAddress', callerIPAddress)
            .withMethod("PUT")
            .go();
    }
    /**
     * Sends a ping to FusionAuth indicating that the user was automatically logged into an application. When using
     * FusionAuth's SSO or your own, you should call this if the user is already logged in centrally, but accesses an
     * application where they no longer have a session. This helps correctly track login counts, times and helps with
     * reporting.
     *
     * @param {LoginPingRequest} request The login request that contains the user credentials used to log them in.
     * @returns {Promise<ClientResponse<LoginResponse>>}
     */
    loginPingWithRequest(request) {
        return this.start()
            .withUri('/api/login')
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * The Logout API is intended to be used to remove the refresh token and access token cookies if they exist on the
     * client and revoke the refresh token stored. This API does nothing if the request does not contain an access
     * token or refresh token cookies.
     *
     * @param {boolean} global When this value is set to true all the refresh tokens issued to the owner of the
     *    provided token will be revoked.
     * @param {string} refreshToken (Optional) The refresh_token as a request parameter instead of coming in via a cookie.
     *    If provided this takes precedence over the cookie.
     * @returns {Promise<ClientResponse<void>>}
     */
    logout(global, refreshToken) {
        return this.startAnonymous()
            .withHeader('Content-Type', 'text/plain')
            .withUri('/api/logout')
            .withParameter('global', global)
            .withParameter('refreshToken', refreshToken)
            .withMethod("POST")
            .go();
    }
    /**
     * The Logout API is intended to be used to remove the refresh token and access token cookies if they exist on the
     * client and revoke the refresh token stored. This API takes the refresh token in the JSON body.
     *
     * @param {LogoutRequest} request The request object that contains all the information used to logout the user.
     * @returns {Promise<ClientResponse<void>>}
     */
    logoutWithRequest(request) {
        return this.startAnonymous()
            .withUri('/api/logout')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Retrieves the identity provider for the given domain. A 200 response code indicates the domain is managed
     * by a registered identity provider. A 404 indicates the domain is not managed.
     *
     * @param {string} domain The domain or email address to lookup.
     * @returns {Promise<ClientResponse<LookupResponse>>}
     */
    lookupIdentityProvider(domain) {
        return this.start()
            .withUri('/api/identity-provider/lookup')
            .withParameter('domain', domain)
            .withMethod("GET")
            .go();
    }
    /**
     * Modifies a temporal user action by changing the expiration of the action and optionally adding a comment to the
     * action.
     *
     * @param {UUID} actionId The Id of the action to modify. This is technically the user action log Id.
     * @param {ActionRequest} request The request that contains all the information about the modification.
     * @returns {Promise<ClientResponse<ActionResponse>>}
     */
    modifyAction(actionId, request) {
        return this.start()
            .withUri('/api/user/action')
            .withUriSegment(actionId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Complete a login request using a passwordless code
     *
     * @param {PasswordlessLoginRequest} request The passwordless login request that contains all the information used to complete login.
     * @returns {Promise<ClientResponse<LoginResponse>>}
     */
    passwordlessLogin(request) {
        return this.startAnonymous()
            .withUri('/api/passwordless/login')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Updates an API key with the given Id.
     *
     * @param {UUID} keyId The Id of the API key. If not provided a secure random api key will be generated.
     * @param {APIKeyRequest} request The request object that contains all the information needed to create the API key.
     * @returns {Promise<ClientResponse<APIKeyResponse>>}
     */
    patchAPIKey(keyId, request) {
        return this.start()
            .withUri('/api/api-key')
            .withUriSegment(keyId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the application with the given Id.
     *
     * @param {UUID} applicationId The Id of the application to update.
     * @param {ApplicationRequest} request The request that contains just the new application information.
     * @returns {Promise<ClientResponse<ApplicationResponse>>}
     */
    patchApplication(applicationId, request) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the application role with the given Id for the application.
     *
     * @param {UUID} applicationId The Id of the application that the role belongs to.
     * @param {UUID} roleId The Id of the role to update.
     * @param {ApplicationRequest} request The request that contains just the new role information.
     * @returns {Promise<ClientResponse<ApplicationResponse>>}
     */
    patchApplicationRole(applicationId, roleId, request) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withUriSegment("role")
            .withUriSegment(roleId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the connector with the given Id.
     *
     * @param {UUID} connectorId The Id of the connector to update.
     * @param {ConnectorRequest} request The request that contains just the new connector information.
     * @returns {Promise<ClientResponse<ConnectorResponse>>}
     */
    patchConnector(connectorId, request) {
        return this.start()
            .withUri('/api/connector')
            .withUriSegment(connectorId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the consent with the given Id.
     *
     * @param {UUID} consentId The Id of the consent to update.
     * @param {ConsentRequest} request The request that contains just the new consent information.
     * @returns {Promise<ClientResponse<ConsentResponse>>}
     */
    patchConsent(consentId, request) {
        return this.start()
            .withUri('/api/consent')
            .withUriSegment(consentId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the email template with the given Id.
     *
     * @param {UUID} emailTemplateId The Id of the email template to update.
     * @param {EmailTemplateRequest} request The request that contains just the new email template information.
     * @returns {Promise<ClientResponse<EmailTemplateResponse>>}
     */
    patchEmailTemplate(emailTemplateId, request) {
        return this.start()
            .withUri('/api/email/template')
            .withUriSegment(emailTemplateId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the Entity with the given Id.
     *
     * @param {UUID} entityId The Id of the Entity Type to update.
     * @param {EntityRequest} request The request that contains just the new Entity information.
     * @returns {Promise<ClientResponse<EntityResponse>>}
     */
    patchEntity(entityId, request) {
        return this.start()
            .withUri('/api/entity')
            .withUriSegment(entityId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the Entity Type with the given Id.
     *
     * @param {UUID} entityTypeId The Id of the Entity Type to update.
     * @param {EntityTypeRequest} request The request that contains just the new Entity Type information.
     * @returns {Promise<ClientResponse<EntityTypeResponse>>}
     */
    patchEntityType(entityTypeId, request) {
        return this.start()
            .withUri('/api/entity/type')
            .withUriSegment(entityTypeId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Patches the permission with the given Id for the entity type.
     *
     * @param {UUID} entityTypeId The Id of the entityType that the permission belongs to.
     * @param {UUID} permissionId The Id of the permission to patch.
     * @param {EntityTypeRequest} request The request that contains the new permission information.
     * @returns {Promise<ClientResponse<EntityTypeResponse>>}
     */
    patchEntityTypePermission(entityTypeId, permissionId, request) {
        return this.start()
            .withUri('/api/entity/type')
            .withUriSegment(entityTypeId)
            .withUriSegment("permission")
            .withUriSegment(permissionId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Patches the form with the given Id.
     *
     * @param {UUID} formId The Id of the form to patch.
     * @param {FormRequest} request The request object that contains the new form information.
     * @returns {Promise<ClientResponse<FormResponse>>}
     */
    patchForm(formId, request) {
        return this.start()
            .withUri('/api/form')
            .withUriSegment(formId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Patches the form field with the given Id.
     *
     * @param {UUID} fieldId The Id of the form field to patch.
     * @param {FormFieldRequest} request The request object that contains the new form field information.
     * @returns {Promise<ClientResponse<FormFieldResponse>>}
     */
    patchFormField(fieldId, request) {
        return this.start()
            .withUri('/api/form/field')
            .withUriSegment(fieldId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the group with the given Id.
     *
     * @param {UUID} groupId The Id of the group to update.
     * @param {GroupRequest} request The request that contains just the new group information.
     * @returns {Promise<ClientResponse<GroupResponse>>}
     */
    patchGroup(groupId, request) {
        return this.start()
            .withUri('/api/group')
            .withUriSegment(groupId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Update the IP Access Control List with the given Id.
     *
     * @param {UUID} accessControlListId The Id of the IP Access Control List to patch.
     * @param {IPAccessControlListRequest} request The request that contains the new IP Access Control List information.
     * @returns {Promise<ClientResponse<IPAccessControlListResponse>>}
     */
    patchIPAccessControlList(accessControlListId, request) {
        return this.start()
            .withUri('/api/ip-acl')
            .withUriSegment(accessControlListId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the identity provider with the given Id.
     *
     * @param {UUID} identityProviderId The Id of the identity provider to update.
     * @param {IdentityProviderRequest} request The request object that contains just the updated identity provider information.
     * @returns {Promise<ClientResponse<IdentityProviderResponse>>}
     */
    patchIdentityProvider(identityProviderId, request) {
        return this.start()
            .withUri('/api/identity-provider')
            .withUriSegment(identityProviderId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the available integrations.
     *
     * @param {IntegrationRequest} request The request that contains just the new integration information.
     * @returns {Promise<ClientResponse<IntegrationResponse>>}
     */
    patchIntegrations(request) {
        return this.start()
            .withUri('/api/integration')
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the lambda with the given Id.
     *
     * @param {UUID} lambdaId The Id of the lambda to update.
     * @param {LambdaRequest} request The request that contains just the new lambda information.
     * @returns {Promise<ClientResponse<LambdaResponse>>}
     */
    patchLambda(lambdaId, request) {
        return this.start()
            .withUri('/api/lambda')
            .withUriSegment(lambdaId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the message template with the given Id.
     *
     * @param {UUID} messageTemplateId The Id of the message template to update.
     * @param {MessageTemplateRequest} request The request that contains just the new message template information.
     * @returns {Promise<ClientResponse<MessageTemplateResponse>>}
     */
    patchMessageTemplate(messageTemplateId, request) {
        return this.start()
            .withUri('/api/message/template')
            .withUriSegment(messageTemplateId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the messenger with the given Id.
     *
     * @param {UUID} messengerId The Id of the messenger to update.
     * @param {MessengerRequest} request The request that contains just the new messenger information.
     * @returns {Promise<ClientResponse<MessengerResponse>>}
     */
    patchMessenger(messengerId, request) {
        return this.start()
            .withUri('/api/messenger')
            .withUriSegment(messengerId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the custom OAuth scope with the given Id for the application.
     *
     * @param {UUID} applicationId The Id of the application that the OAuth scope belongs to.
     * @param {UUID} scopeId The Id of the OAuth scope to update.
     * @param {ApplicationOAuthScopeRequest} request The request that contains just the new OAuth scope information.
     * @returns {Promise<ClientResponse<ApplicationOAuthScopeResponse>>}
     */
    patchOAuthScope(applicationId, scopeId, request) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withUriSegment("scope")
            .withUriSegment(scopeId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the registration for the user with the given Id and the application defined in the request.
     *
     * @param {UUID} userId The Id of the user whose registration is going to be updated.
     * @param {RegistrationRequest} request The request that contains just the new registration information.
     * @returns {Promise<ClientResponse<RegistrationResponse>>}
     */
    patchRegistration(userId, request) {
        return this.start()
            .withUri('/api/user/registration')
            .withUriSegment(userId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the system configuration.
     *
     * @param {SystemConfigurationRequest} request The request that contains just the new system configuration information.
     * @returns {Promise<ClientResponse<SystemConfigurationResponse>>}
     */
    patchSystemConfiguration(request) {
        return this.start()
            .withUri('/api/system-configuration')
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the tenant with the given Id.
     *
     * @param {UUID} tenantId The Id of the tenant to update.
     * @param {TenantRequest} request The request that contains just the new tenant information.
     * @returns {Promise<ClientResponse<TenantResponse>>}
     */
    patchTenant(tenantId, request) {
        return this.start()
            .withUri('/api/tenant')
            .withUriSegment(tenantId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the theme with the given Id.
     *
     * @param {UUID} themeId The Id of the theme to update.
     * @param {ThemeRequest} request The request that contains just the new theme information.
     * @returns {Promise<ClientResponse<ThemeResponse>>}
     */
    patchTheme(themeId, request) {
        return this.start()
            .withUri('/api/theme')
            .withUriSegment(themeId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the user with the given Id.
     *
     * @param {UUID} userId The Id of the user to update.
     * @param {UserRequest} request The request that contains just the new user information.
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    patchUser(userId, request) {
        return this.start()
            .withUri('/api/user')
            .withUriSegment(userId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the user action with the given Id.
     *
     * @param {UUID} userActionId The Id of the user action to update.
     * @param {UserActionRequest} request The request that contains just the new user action information.
     * @returns {Promise<ClientResponse<UserActionResponse>>}
     */
    patchUserAction(userActionId, request) {
        return this.start()
            .withUri('/api/user-action')
            .withUriSegment(userActionId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, the user action reason with the given Id.
     *
     * @param {UUID} userActionReasonId The Id of the user action reason to update.
     * @param {UserActionReasonRequest} request The request that contains just the new user action reason information.
     * @returns {Promise<ClientResponse<UserActionReasonResponse>>}
     */
    patchUserActionReason(userActionReasonId, request) {
        return this.start()
            .withUri('/api/user-action-reason')
            .withUriSegment(userActionReasonId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Updates, via PATCH, a single User consent by Id.
     *
     * @param {UUID} userConsentId The User Consent Id
     * @param {UserConsentRequest} request The request that contains just the new user consent information.
     * @returns {Promise<ClientResponse<UserConsentResponse>>}
     */
    patchUserConsent(userConsentId, request) {
        return this.start()
            .withUri('/api/user/consent')
            .withUriSegment(userConsentId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Patches the webhook with the given Id.
     *
     * @param {UUID} webhookId The Id of the webhook to update.
     * @param {WebhookRequest} request The request that contains the new webhook information.
     * @returns {Promise<ClientResponse<WebhookResponse>>}
     */
    patchWebhook(webhookId, request) {
        return this.start()
            .withUri('/api/webhook')
            .withUriSegment(webhookId)
            .withJSONBody(request)
            .withMethod("PATCH")
            .go();
    }
    /**
     * Reactivates the application with the given Id.
     *
     * @param {UUID} applicationId The Id of the application to reactivate.
     * @returns {Promise<ClientResponse<ApplicationResponse>>}
     */
    reactivateApplication(applicationId) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withParameter('reactivate', true)
            .withMethod("PUT")
            .go();
    }
    /**
     * Reactivates the user with the given Id.
     *
     * @param {UUID} userId The Id of the user to reactivate.
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    reactivateUser(userId) {
        return this.start()
            .withUri('/api/user')
            .withUriSegment(userId)
            .withParameter('reactivate', true)
            .withMethod("PUT")
            .go();
    }
    /**
     * Reactivates the user action with the given Id.
     *
     * @param {UUID} userActionId The Id of the user action to reactivate.
     * @returns {Promise<ClientResponse<UserActionResponse>>}
     */
    reactivateUserAction(userActionId) {
        return this.start()
            .withUri('/api/user-action')
            .withUriSegment(userActionId)
            .withParameter('reactivate', true)
            .withMethod("PUT")
            .go();
    }
    /**
     * Reconcile a User to FusionAuth using JWT issued from another Identity Provider.
     *
     * @param {IdentityProviderLoginRequest} request The reconcile request that contains the data to reconcile the User.
     * @returns {Promise<ClientResponse<LoginResponse>>}
     */
    reconcileJWT(request) {
        return this.startAnonymous()
            .withUri('/api/jwt/reconcile')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Request a refresh of the Entity search index. This API is not generally necessary and the search index will become consistent in a
     * reasonable amount of time. There may be scenarios where you may wish to manually request an index refresh. One example may be
     * if you are using the Search API or Delete Tenant API immediately following a Entity Create etc, you may wish to request a refresh to
     *  ensure the index immediately current before making a query request to the search index.
     *
     * @returns {Promise<ClientResponse<void>>}
     */
    refreshEntitySearchIndex() {
        return this.start()
            .withUri('/api/entity/search')
            .withMethod("PUT")
            .go();
    }
    /**
     * Request a refresh of the User search index. This API is not generally necessary and the search index will become consistent in a
     * reasonable amount of time. There may be scenarios where you may wish to manually request an index refresh. One example may be
     * if you are using the Search API or Delete Tenant API immediately following a User Create etc, you may wish to request a refresh to
     *  ensure the index immediately current before making a query request to the search index.
     *
     * @returns {Promise<ClientResponse<void>>}
     */
    refreshUserSearchIndex() {
        return this.start()
            .withUri('/api/user/search')
            .withMethod("PUT")
            .go();
    }
    /**
     * Regenerates any keys that are used by the FusionAuth Reactor.
     *
     * @returns {Promise<ClientResponse<void>>}
     */
    regenerateReactorKeys() {
        return this.start()
            .withUri('/api/reactor')
            .withMethod("PUT")
            .go();
    }
    /**
     * Registers a user for an application. If you provide the User and the UserRegistration object on this request, it
     * will create the user as well as register them for the application. This is called a Full Registration. However, if
     * you only provide the UserRegistration object, then the user must already exist and they will be registered for the
     * application. The user Id can also be provided and it will either be used to look up an existing user or it will be
     * used for the newly created User.
     *
     * @param {UUID} userId (Optional) The Id of the user being registered for the application and optionally created.
     * @param {RegistrationRequest} request The request that optionally contains the User and must contain the UserRegistration.
     * @returns {Promise<ClientResponse<RegistrationResponse>>}
     */
    register(userId, request) {
        return this.start()
            .withUri('/api/user/registration')
            .withUriSegment(userId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Requests Elasticsearch to delete and rebuild the index for FusionAuth users or entities. Be very careful when running this request as it will
     * increase the CPU and I/O load on your database until the operation completes. Generally speaking you do not ever need to run this operation unless
     * instructed by FusionAuth support, or if you are migrating a database another system and you are not brining along the Elasticsearch index.
     *
     * You have been warned.
     *
     * @param {ReindexRequest} request The request that contains the index name.
     * @returns {Promise<ClientResponse<void>>}
     */
    reindex(request) {
        return this.start()
            .withUri('/api/system/reindex')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Removes a user from the family with the given Id.
     *
     * @param {UUID} familyId The Id of the family to remove the user from.
     * @param {UUID} userId The Id of the user to remove from the family.
     * @returns {Promise<ClientResponse<void>>}
     */
    removeUserFromFamily(familyId, userId) {
        return this.start()
            .withUri('/api/user/family')
            .withUriSegment(familyId)
            .withUriSegment(userId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Re-sends the verification email to the user.
     *
     * @param {string} email The email address of the user that needs a new verification email.
     * @returns {Promise<ClientResponse<VerifyEmailResponse>>}
     */
    resendEmailVerification(email) {
        return this.start()
            .withUri('/api/user/verify-email')
            .withParameter('email', email)
            .withMethod("PUT")
            .go();
    }
    /**
     * Re-sends the verification email to the user. If the Application has configured a specific email template this will be used
     * instead of the tenant configuration.
     *
     * @param {UUID} applicationId The unique Application Id to used to resolve an application specific email template.
     * @param {string} email The email address of the user that needs a new verification email.
     * @returns {Promise<ClientResponse<VerifyEmailResponse>>}
     */
    resendEmailVerificationWithApplicationTemplate(applicationId, email) {
        return this.start()
            .withUri('/api/user/verify-email')
            .withParameter('applicationId', applicationId)
            .withParameter('email', email)
            .withMethod("PUT")
            .go();
    }
    /**
     * Re-sends the application registration verification email to the user.
     *
     * @param {string} email The email address of the user that needs a new verification email.
     * @param {UUID} applicationId The Id of the application to be verified.
     * @returns {Promise<ClientResponse<VerifyRegistrationResponse>>}
     */
    resendRegistrationVerification(email, applicationId) {
        return this.start()
            .withUri('/api/user/verify-registration')
            .withParameter('email', email)
            .withParameter('applicationId', applicationId)
            .withMethod("PUT")
            .go();
    }
    /**
     * Retrieves an authentication API key for the given Id.
     *
     * @param {UUID} keyId The Id of the API key to retrieve.
     * @returns {Promise<ClientResponse<APIKeyResponse>>}
     */
    retrieveAPIKey(keyId) {
        return this.start()
            .withUri('/api/api-key')
            .withUriSegment(keyId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves a single action log (the log of a user action that was taken on a user previously) for the given Id.
     *
     * @param {UUID} actionId The Id of the action to retrieve.
     * @returns {Promise<ClientResponse<ActionResponse>>}
     */
    retrieveAction(actionId) {
        return this.start()
            .withUri('/api/user/action')
            .withUriSegment(actionId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the actions for the user with the given Id. This will return all time based actions that are active,
     * and inactive as well as non-time based actions.
     *
     * @param {UUID} userId The Id of the user to fetch the actions for.
     * @returns {Promise<ClientResponse<ActionResponse>>}
     */
    retrieveActions(userId) {
        return this.start()
            .withUri('/api/user/action')
            .withParameter('userId', userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the actions for the user with the given Id that are currently preventing the User from logging in.
     *
     * @param {UUID} userId The Id of the user to fetch the actions for.
     * @returns {Promise<ClientResponse<ActionResponse>>}
     */
    retrieveActionsPreventingLogin(userId) {
        return this.start()
            .withUri('/api/user/action')
            .withParameter('userId', userId)
            .withParameter('preventingLogin', true)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the actions for the user with the given Id that are currently active.
     * An active action means one that is time based and has not been canceled, and has not ended.
     *
     * @param {UUID} userId The Id of the user to fetch the actions for.
     * @returns {Promise<ClientResponse<ActionResponse>>}
     */
    retrieveActiveActions(userId) {
        return this.start()
            .withUri('/api/user/action')
            .withParameter('userId', userId)
            .withParameter('active', true)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the application for the given Id or all the applications if the Id is null.
     *
     * @param {UUID} applicationId (Optional) The application Id.
     * @returns {Promise<ClientResponse<ApplicationResponse>>}
     */
    retrieveApplication(applicationId) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the applications.
     *
     * @returns {Promise<ClientResponse<ApplicationResponse>>}
     */
    retrieveApplications() {
        return this.start()
            .withUri('/api/application')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves a single audit log for the given Id.
     *
     * @param {number} auditLogId The Id of the audit log to retrieve.
     * @returns {Promise<ClientResponse<AuditLogResponse>>}
     */
    retrieveAuditLog(auditLogId) {
        return this.start()
            .withUri('/api/system/audit-log')
            .withUriSegment(auditLogId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the connector with the given Id.
     *
     * @param {UUID} connectorId The Id of the connector.
     * @returns {Promise<ClientResponse<ConnectorResponse>>}
     */
    retrieveConnector(connectorId) {
        return this.start()
            .withUri('/api/connector')
            .withUriSegment(connectorId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the connectors.
     *
     * @returns {Promise<ClientResponse<ConnectorResponse>>}
     */
    retrieveConnectors() {
        return this.start()
            .withUri('/api/connector')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the Consent for the given Id.
     *
     * @param {UUID} consentId The Id of the consent.
     * @returns {Promise<ClientResponse<ConsentResponse>>}
     */
    retrieveConsent(consentId) {
        return this.start()
            .withUri('/api/consent')
            .withUriSegment(consentId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the consent.
     *
     * @returns {Promise<ClientResponse<ConsentResponse>>}
     */
    retrieveConsents() {
        return this.start()
            .withUri('/api/consent')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the daily active user report between the two instants. If you specify an application Id, it will only
     * return the daily active counts for that application.
     *
     * @param {UUID} applicationId (Optional) The application Id.
     * @param {number} start The start instant as UTC milliseconds since Epoch.
     * @param {number} end The end instant as UTC milliseconds since Epoch.
     * @returns {Promise<ClientResponse<DailyActiveUserReportResponse>>}
     */
    retrieveDailyActiveReport(applicationId, start, end) {
        return this.start()
            .withUri('/api/report/daily-active-user')
            .withParameter('applicationId', applicationId)
            .withParameter('start', start)
            .withParameter('end', end)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the email template for the given Id. If you don't specify the Id, this will return all the email templates.
     *
     * @param {UUID} emailTemplateId (Optional) The Id of the email template.
     * @returns {Promise<ClientResponse<EmailTemplateResponse>>}
     */
    retrieveEmailTemplate(emailTemplateId) {
        return this.start()
            .withUri('/api/email/template')
            .withUriSegment(emailTemplateId)
            .withMethod("GET")
            .go();
    }
    /**
     * Creates a preview of the email template provided in the request. This allows you to preview an email template that
     * hasn't been saved to the database yet. The entire email template does not need to be provided on the request. This
     * will create the preview based on whatever is given.
     *
     * @param {PreviewRequest} request The request that contains the email template and optionally a locale to render it in.
     * @returns {Promise<ClientResponse<PreviewResponse>>}
     */
    retrieveEmailTemplatePreview(request) {
        return this.start()
            .withUri('/api/email/template/preview')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Retrieves all the email templates.
     *
     * @returns {Promise<ClientResponse<EmailTemplateResponse>>}
     */
    retrieveEmailTemplates() {
        return this.start()
            .withUri('/api/email/template')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the Entity for the given Id.
     *
     * @param {UUID} entityId The Id of the Entity.
     * @returns {Promise<ClientResponse<EntityResponse>>}
     */
    retrieveEntity(entityId) {
        return this.start()
            .withUri('/api/entity')
            .withUriSegment(entityId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves an Entity Grant for the given Entity and User/Entity.
     *
     * @param {UUID} entityId The Id of the Entity.
     * @param {UUID} recipientEntityId (Optional) The Id of the Entity that the Entity Grant is for.
     * @param {UUID} userId (Optional) The Id of the User that the Entity Grant is for.
     * @returns {Promise<ClientResponse<EntityGrantResponse>>}
     */
    retrieveEntityGrant(entityId, recipientEntityId, userId) {
        return this.start()
            .withUri('/api/entity')
            .withUriSegment(entityId)
            .withUriSegment("grant")
            .withParameter('recipientEntityId', recipientEntityId)
            .withParameter('userId', userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the Entity Type for the given Id.
     *
     * @param {UUID} entityTypeId The Id of the Entity Type.
     * @returns {Promise<ClientResponse<EntityTypeResponse>>}
     */
    retrieveEntityType(entityTypeId) {
        return this.start()
            .withUri('/api/entity/type')
            .withUriSegment(entityTypeId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the Entity Types.
     *
     * @returns {Promise<ClientResponse<EntityTypeResponse>>}
     */
    retrieveEntityTypes() {
        return this.start()
            .withUri('/api/entity/type')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves a single event log for the given Id.
     *
     * @param {number} eventLogId The Id of the event log to retrieve.
     * @returns {Promise<ClientResponse<EventLogResponse>>}
     */
    retrieveEventLog(eventLogId) {
        return this.start()
            .withUri('/api/system/event-log')
            .withUriSegment(eventLogId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the families that a user belongs to.
     *
     * @param {UUID} userId The User's id
     * @returns {Promise<ClientResponse<FamilyResponse>>}
     */
    retrieveFamilies(userId) {
        return this.start()
            .withUri('/api/user/family')
            .withParameter('userId', userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the members of a family by the unique Family Id.
     *
     * @param {UUID} familyId The unique Id of the Family.
     * @returns {Promise<ClientResponse<FamilyResponse>>}
     */
    retrieveFamilyMembersByFamilyId(familyId) {
        return this.start()
            .withUri('/api/user/family')
            .withUriSegment(familyId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the form with the given Id.
     *
     * @param {UUID} formId The Id of the form.
     * @returns {Promise<ClientResponse<FormResponse>>}
     */
    retrieveForm(formId) {
        return this.start()
            .withUri('/api/form')
            .withUriSegment(formId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the form field with the given Id.
     *
     * @param {UUID} fieldId The Id of the form field.
     * @returns {Promise<ClientResponse<FormFieldResponse>>}
     */
    retrieveFormField(fieldId) {
        return this.start()
            .withUri('/api/form/field')
            .withUriSegment(fieldId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the forms fields
     *
     * @returns {Promise<ClientResponse<FormFieldResponse>>}
     */
    retrieveFormFields() {
        return this.start()
            .withUri('/api/form/field')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the forms.
     *
     * @returns {Promise<ClientResponse<FormResponse>>}
     */
    retrieveForms() {
        return this.start()
            .withUri('/api/form')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the group for the given Id.
     *
     * @param {UUID} groupId The Id of the group.
     * @returns {Promise<ClientResponse<GroupResponse>>}
     */
    retrieveGroup(groupId) {
        return this.start()
            .withUri('/api/group')
            .withUriSegment(groupId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the groups.
     *
     * @returns {Promise<ClientResponse<GroupResponse>>}
     */
    retrieveGroups() {
        return this.start()
            .withUri('/api/group')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the IP Access Control List with the given Id.
     *
     * @param {UUID} ipAccessControlListId The Id of the IP Access Control List.
     * @returns {Promise<ClientResponse<IPAccessControlListResponse>>}
     */
    retrieveIPAccessControlList(ipAccessControlListId) {
        return this.start()
            .withUri('/api/ip-acl')
            .withUriSegment(ipAccessControlListId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the identity provider for the given Id or all the identity providers if the Id is null.
     *
     * @param {UUID} identityProviderId The identity provider Id.
     * @returns {Promise<ClientResponse<IdentityProviderResponse>>}
     */
    retrieveIdentityProvider(identityProviderId) {
        return this.start()
            .withUri('/api/identity-provider')
            .withUriSegment(identityProviderId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves one or more identity provider for the given type. For types such as Google, Facebook, Twitter and LinkedIn, only a single
     * identity provider can exist. For types such as OpenID Connect and SAMLv2 more than one identity provider can be configured so this request
     * may return multiple identity providers.
     *
     * @param {IdentityProviderType} type The type of the identity provider.
     * @returns {Promise<ClientResponse<IdentityProviderResponse>>}
     */
    retrieveIdentityProviderByType(type) {
        return this.start()
            .withUri('/api/identity-provider')
            .withParameter('type', type)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the identity providers.
     *
     * @returns {Promise<ClientResponse<IdentityProviderResponse>>}
     */
    retrieveIdentityProviders() {
        return this.start()
            .withUri('/api/identity-provider')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the actions for the user with the given Id that are currently inactive.
     * An inactive action means one that is time based and has been canceled or has expired, or is not time based.
     *
     * @param {UUID} userId The Id of the user to fetch the actions for.
     * @returns {Promise<ClientResponse<ActionResponse>>}
     */
    retrieveInactiveActions(userId) {
        return this.start()
            .withUri('/api/user/action')
            .withParameter('userId', userId)
            .withParameter('active', false)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the applications that are currently inactive.
     *
     * @returns {Promise<ClientResponse<ApplicationResponse>>}
     */
    retrieveInactiveApplications() {
        return this.start()
            .withUri('/api/application')
            .withParameter('inactive', true)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the user actions that are currently inactive.
     *
     * @returns {Promise<ClientResponse<UserActionResponse>>}
     */
    retrieveInactiveUserActions() {
        return this.start()
            .withUri('/api/user-action')
            .withParameter('inactive', true)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the available integrations.
     *
     * @returns {Promise<ClientResponse<IntegrationResponse>>}
     */
    retrieveIntegration() {
        return this.start()
            .withUri('/api/integration')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the Public Key configured for verifying JSON Web Tokens (JWT) by the key Id (kid).
     *
     * @param {string} keyId The Id of the public key (kid).
     * @returns {Promise<ClientResponse<PublicKeyResponse>>}
     */
    retrieveJWTPublicKey(keyId) {
        return this.startAnonymous()
            .withUri('/api/jwt/public-key')
            .withParameter('kid', keyId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the Public Key configured for verifying the JSON Web Tokens (JWT) issued by the Login API by the Application Id.
     *
     * @param {string} applicationId The Id of the Application for which this key is used.
     * @returns {Promise<ClientResponse<PublicKeyResponse>>}
     */
    retrieveJWTPublicKeyByApplicationId(applicationId) {
        return this.startAnonymous()
            .withUri('/api/jwt/public-key')
            .withParameter('applicationId', applicationId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all Public Keys configured for verifying JSON Web Tokens (JWT).
     *
     * @returns {Promise<ClientResponse<PublicKeyResponse>>}
     */
    retrieveJWTPublicKeys() {
        return this.startAnonymous()
            .withUri('/api/jwt/public-key')
            .withMethod("GET")
            .go();
    }
    /**
     * Returns public keys used by FusionAuth to cryptographically verify JWTs using the JSON Web Key format.
     *
     * @returns {Promise<ClientResponse<JWKSResponse>>}
     */
    retrieveJsonWebKeySet() {
        return this.startAnonymous()
            .withUri('/.well-known/jwks.json')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the key for the given Id.
     *
     * @param {UUID} keyId The Id of the key.
     * @returns {Promise<ClientResponse<KeyResponse>>}
     */
    retrieveKey(keyId) {
        return this.start()
            .withUri('/api/key')
            .withUriSegment(keyId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the keys.
     *
     * @returns {Promise<ClientResponse<KeyResponse>>}
     */
    retrieveKeys() {
        return this.start()
            .withUri('/api/key')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the lambda for the given Id.
     *
     * @param {UUID} lambdaId The Id of the lambda.
     * @returns {Promise<ClientResponse<LambdaResponse>>}
     */
    retrieveLambda(lambdaId) {
        return this.start()
            .withUri('/api/lambda')
            .withUriSegment(lambdaId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the lambdas.
     *
     * @returns {Promise<ClientResponse<LambdaResponse>>}
     */
    retrieveLambdas() {
        return this.start()
            .withUri('/api/lambda')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the lambdas for the provided type.
     *
     * @param {LambdaType} type The type of the lambda to return.
     * @returns {Promise<ClientResponse<LambdaResponse>>}
     */
    retrieveLambdasByType(type) {
        return this.start()
            .withUri('/api/lambda')
            .withParameter('type', type)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the login report between the two instants. If you specify an application Id, it will only return the
     * login counts for that application.
     *
     * @param {UUID} applicationId (Optional) The application Id.
     * @param {number} start The start instant as UTC milliseconds since Epoch.
     * @param {number} end The end instant as UTC milliseconds since Epoch.
     * @returns {Promise<ClientResponse<LoginReportResponse>>}
     */
    retrieveLoginReport(applicationId, start, end) {
        return this.start()
            .withUri('/api/report/login')
            .withParameter('applicationId', applicationId)
            .withParameter('start', start)
            .withParameter('end', end)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the message template for the given Id. If you don't specify the Id, this will return all the message templates.
     *
     * @param {UUID} messageTemplateId (Optional) The Id of the message template.
     * @returns {Promise<ClientResponse<MessageTemplateResponse>>}
     */
    retrieveMessageTemplate(messageTemplateId) {
        return this.start()
            .withUri('/api/message/template')
            .withUriSegment(messageTemplateId)
            .withMethod("GET")
            .go();
    }
    /**
     * Creates a preview of the message template provided in the request, normalized to a given locale.
     *
     * @param {PreviewMessageTemplateRequest} request The request that contains the email template and optionally a locale to render it in.
     * @returns {Promise<ClientResponse<PreviewMessageTemplateResponse>>}
     */
    retrieveMessageTemplatePreview(request) {
        return this.start()
            .withUri('/api/message/template/preview')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Retrieves all the message templates.
     *
     * @returns {Promise<ClientResponse<MessageTemplateResponse>>}
     */
    retrieveMessageTemplates() {
        return this.start()
            .withUri('/api/message/template')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the messenger with the given Id.
     *
     * @param {UUID} messengerId The Id of the messenger.
     * @returns {Promise<ClientResponse<MessengerResponse>>}
     */
    retrieveMessenger(messengerId) {
        return this.start()
            .withUri('/api/messenger')
            .withUriSegment(messengerId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the messengers.
     *
     * @returns {Promise<ClientResponse<MessengerResponse>>}
     */
    retrieveMessengers() {
        return this.start()
            .withUri('/api/messenger')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the monthly active user report between the two instants. If you specify an application Id, it will only
     * return the monthly active counts for that application.
     *
     * @param {UUID} applicationId (Optional) The application Id.
     * @param {number} start The start instant as UTC milliseconds since Epoch.
     * @param {number} end The end instant as UTC milliseconds since Epoch.
     * @returns {Promise<ClientResponse<MonthlyActiveUserReportResponse>>}
     */
    retrieveMonthlyActiveReport(applicationId, start, end) {
        return this.start()
            .withUri('/api/report/monthly-active-user')
            .withParameter('applicationId', applicationId)
            .withParameter('start', start)
            .withParameter('end', end)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves a custom OAuth scope.
     *
     * @param {UUID} applicationId The Id of the application that the OAuth scope belongs to.
     * @param {UUID} scopeId The Id of the OAuth scope to retrieve.
     * @returns {Promise<ClientResponse<ApplicationOAuthScopeResponse>>}
     */
    retrieveOAuthScope(applicationId, scopeId) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withUriSegment("scope")
            .withUriSegment(scopeId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the Oauth2 configuration for the application for the given Application Id.
     *
     * @param {UUID} applicationId The Id of the Application to retrieve OAuth configuration.
     * @returns {Promise<ClientResponse<OAuthConfigurationResponse>>}
     */
    retrieveOauthConfiguration(applicationId) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withUriSegment("oauth-configuration")
            .withMethod("GET")
            .go();
    }
    /**
     * Returns the well known OpenID Configuration JSON document
     *
     * @returns {Promise<ClientResponse<OpenIdConfiguration>>}
     */
    retrieveOpenIdConfiguration() {
        return this.startAnonymous()
            .withUri('/.well-known/openid-configuration')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the password validation rules for a specific tenant. This method requires a tenantId to be provided
     * through the use of a Tenant scoped API key or an HTTP header X-FusionAuth-TenantId to specify the Tenant Id.
     *
     * This API does not require an API key.
     *
     * @returns {Promise<ClientResponse<PasswordValidationRulesResponse>>}
     */
    retrievePasswordValidationRules() {
        return this.startAnonymous()
            .withUri('/api/tenant/password-validation-rules')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the password validation rules for a specific tenant.
     *
     * This API does not require an API key.
     *
     * @param {UUID} tenantId The Id of the tenant.
     * @returns {Promise<ClientResponse<PasswordValidationRulesResponse>>}
     */
    retrievePasswordValidationRulesWithTenantId(tenantId) {
        return this.startAnonymous()
            .withUri('/api/tenant/password-validation-rules')
            .withUriSegment(tenantId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the children for the given parent email address.
     *
     * @param {string} parentEmail The email of the parent.
     * @returns {Promise<ClientResponse<PendingResponse>>}
     */
    retrievePendingChildren(parentEmail) {
        return this.start()
            .withUri('/api/user/family/pending')
            .withParameter('parentEmail', parentEmail)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieve a pending identity provider link. This is useful to validate a pending link and retrieve meta-data about the identity provider link.
     *
     * @param {string} pendingLinkId The pending link Id.
     * @param {UUID} userId The optional userId. When provided additional meta-data will be provided to identify how many links if any the user already has.
     * @returns {Promise<ClientResponse<IdentityProviderPendingLinkResponse>>}
     */
    retrievePendingLink(pendingLinkId, userId) {
        return this.start()
            .withUri('/api/identity-provider/link/pending')
            .withUriSegment(pendingLinkId)
            .withParameter('userId', userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the FusionAuth Reactor metrics.
     *
     * @returns {Promise<ClientResponse<ReactorMetricsResponse>>}
     */
    retrieveReactorMetrics() {
        return this.start()
            .withUri('/api/reactor/metrics')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the FusionAuth Reactor status.
     *
     * @returns {Promise<ClientResponse<ReactorResponse>>}
     */
    retrieveReactorStatus() {
        return this.start()
            .withUri('/api/reactor')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the last number of login records.
     *
     * @param {number} offset The initial record. e.g. 0 is the last login, 100 will be the 100th most recent login.
     * @param {number} limit (Optional, defaults to 10) The number of records to retrieve.
     * @returns {Promise<ClientResponse<RecentLoginResponse>>}
     */
    retrieveRecentLogins(offset, limit) {
        return this.start()
            .withUri('/api/user/recent-login')
            .withParameter('offset', offset)
            .withParameter('limit', limit)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves a single refresh token by unique Id. This is not the same thing as the string value of the refresh token. If you have that, you already have what you need.
     *
     * @param {UUID} tokenId The Id of the token.
     * @returns {Promise<ClientResponse<RefreshTokenResponse>>}
     */
    retrieveRefreshTokenById(tokenId) {
        return this.start()
            .withUri('/api/jwt/refresh')
            .withUriSegment(tokenId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the refresh tokens that belong to the user with the given Id.
     *
     * @param {UUID} userId The Id of the user.
     * @returns {Promise<ClientResponse<RefreshTokenResponse>>}
     */
    retrieveRefreshTokens(userId) {
        return this.start()
            .withUri('/api/jwt/refresh')
            .withParameter('userId', userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the user registration for the user with the given Id and the given application Id.
     *
     * @param {UUID} userId The Id of the user.
     * @param {UUID} applicationId The Id of the application.
     * @returns {Promise<ClientResponse<RegistrationResponse>>}
     */
    retrieveRegistration(userId, applicationId) {
        return this.start()
            .withUri('/api/user/registration')
            .withUriSegment(userId)
            .withUriSegment(applicationId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the registration report between the two instants. If you specify an application Id, it will only return
     * the registration counts for that application.
     *
     * @param {UUID} applicationId (Optional) The application Id.
     * @param {number} start The start instant as UTC milliseconds since Epoch.
     * @param {number} end The end instant as UTC milliseconds since Epoch.
     * @returns {Promise<ClientResponse<RegistrationReportResponse>>}
     */
    retrieveRegistrationReport(applicationId, start, end) {
        return this.start()
            .withUri('/api/report/registration')
            .withParameter('applicationId', applicationId)
            .withParameter('start', start)
            .withParameter('end', end)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieve the status of a re-index process. A status code of 200 indicates the re-index is in progress, a status code of
     * 404 indicates no re-index is in progress.
     *
     * @returns {Promise<ClientResponse<void>>}
     */
    retrieveReindexStatus() {
        return this.start()
            .withUri('/api/system/reindex')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the system configuration.
     *
     * @returns {Promise<ClientResponse<SystemConfigurationResponse>>}
     */
    retrieveSystemConfiguration() {
        return this.start()
            .withUri('/api/system-configuration')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the FusionAuth system health. This API will return 200 if the system is healthy, and 500 if the system is un-healthy.
     *
     * @returns {Promise<ClientResponse<void>>}
     */
    retrieveSystemHealth() {
        return this.startAnonymous()
            .withUri('/api/health')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the FusionAuth system status. This request is anonymous and does not require an API key. When an API key is not provided the response will contain a single value in the JSON response indicating the current health check.
     *
     * @returns {Promise<ClientResponse<StatusResponse>>}
     */
    retrieveSystemStatus() {
        return this.startAnonymous()
            .withUri('/api/status')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the FusionAuth system status using an API key. Using an API key will cause the response to include the product version, health checks and various runtime metrics.
     *
     * @returns {Promise<ClientResponse<StatusResponse>>}
     */
    retrieveSystemStatusUsingAPIKey() {
        return this.start()
            .withUri('/api/status')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the tenant for the given Id.
     *
     * @param {UUID} tenantId The Id of the tenant.
     * @returns {Promise<ClientResponse<TenantResponse>>}
     */
    retrieveTenant(tenantId) {
        return this.start()
            .withUri('/api/tenant')
            .withUriSegment(tenantId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the tenants.
     *
     * @returns {Promise<ClientResponse<TenantResponse>>}
     */
    retrieveTenants() {
        return this.start()
            .withUri('/api/tenant')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the theme for the given Id.
     *
     * @param {UUID} themeId The Id of the theme.
     * @returns {Promise<ClientResponse<ThemeResponse>>}
     */
    retrieveTheme(themeId) {
        return this.start()
            .withUri('/api/theme')
            .withUriSegment(themeId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the themes.
     *
     * @returns {Promise<ClientResponse<ThemeResponse>>}
     */
    retrieveThemes() {
        return this.start()
            .withUri('/api/theme')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the totals report. This contains all the total counts for each application and the global registration
     * count.
     *
     * @returns {Promise<ClientResponse<TotalsReportResponse>>}
     */
    retrieveTotalReport() {
        return this.start()
            .withUri('/api/report/totals')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieve two-factor recovery codes for a user.
     *
     * @param {UUID} userId The Id of the user to retrieve Two Factor recovery codes.
     * @returns {Promise<ClientResponse<TwoFactorRecoveryCodeResponse>>}
     */
    retrieveTwoFactorRecoveryCodes(userId) {
        return this.start()
            .withUri('/api/user/two-factor/recovery-code')
            .withUriSegment(userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieve a user's two-factor status.
     *
     * This can be used to see if a user will need to complete a two-factor challenge to complete a login,
     * and optionally identify the state of the two-factor trust across various applications.
     *
     * @param {UUID} userId The user Id to retrieve the Two-Factor status.
     * @param {UUID} applicationId The optional applicationId to verify.
     * @param {string} twoFactorTrustId The optional two-factor trust Id to verify.
     * @returns {Promise<ClientResponse<TwoFactorStatusResponse>>}
     */
    retrieveTwoFactorStatus(userId, applicationId, twoFactorTrustId) {
        return this.start()
            .withUri('/api/two-factor/status')
            .withParameter('userId', userId)
            .withParameter('applicationId', applicationId)
            .withUriSegment(twoFactorTrustId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the user for the given Id.
     *
     * @param {UUID} userId The Id of the user.
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    retrieveUser(userId) {
        return this.start()
            .withUri('/api/user')
            .withUriSegment(userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the user action for the given Id. If you pass in null for the Id, this will return all the user
     * actions.
     *
     * @param {UUID} userActionId (Optional) The Id of the user action.
     * @returns {Promise<ClientResponse<UserActionResponse>>}
     */
    retrieveUserAction(userActionId) {
        return this.start()
            .withUri('/api/user-action')
            .withUriSegment(userActionId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the user action reason for the given Id. If you pass in null for the Id, this will return all the user
     * action reasons.
     *
     * @param {UUID} userActionReasonId (Optional) The Id of the user action reason.
     * @returns {Promise<ClientResponse<UserActionReasonResponse>>}
     */
    retrieveUserActionReason(userActionReasonId) {
        return this.start()
            .withUri('/api/user-action-reason')
            .withUriSegment(userActionReasonId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the user action reasons.
     *
     * @returns {Promise<ClientResponse<UserActionReasonResponse>>}
     */
    retrieveUserActionReasons() {
        return this.start()
            .withUri('/api/user-action-reason')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the user actions.
     *
     * @returns {Promise<ClientResponse<UserActionResponse>>}
     */
    retrieveUserActions() {
        return this.start()
            .withUri('/api/user-action')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the user by a change password Id. The intended use of this API is to retrieve a user after the forgot
     * password workflow has been initiated and you may not know the user's email or username.
     *
     * @param {string} changePasswordId The unique change password Id that was sent via email or returned by the Forgot Password API.
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    retrieveUserByChangePasswordId(changePasswordId) {
        return this.start()
            .withUri('/api/user')
            .withParameter('changePasswordId', changePasswordId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the user for the given email.
     *
     * @param {string} email The email of the user.
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    retrieveUserByEmail(email) {
        return this.start()
            .withUri('/api/user')
            .withParameter('email', email)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the user for the loginId. The loginId can be either the username or the email.
     *
     * @param {string} loginId The email or username of the user.
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    retrieveUserByLoginId(loginId) {
        return this.start()
            .withUri('/api/user')
            .withParameter('loginId', loginId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the user for the loginId, using specific loginIdTypes.
     *
     * @param {string} loginId The email or username of the user.
     * @param {Array<String>} loginIdTypes The identity types that FusionAuth will compare the loginId to.
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    retrieveUserByLoginIdWithLoginIdTypes(loginId, loginIdTypes) {
        return this.start()
            .withUri('/api/user')
            .withParameter('loginId', loginId)
            .withParameter('loginIdTypes', loginIdTypes)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the user for the given username.
     *
     * @param {string} username The username of the user.
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    retrieveUserByUsername(username) {
        return this.start()
            .withUri('/api/user')
            .withParameter('username', username)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the user by a verificationId. The intended use of this API is to retrieve a user after the forgot
     * password workflow has been initiated and you may not know the user's email or username.
     *
     * @param {string} verificationId The unique verification Id that has been set on the user object.
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    retrieveUserByVerificationId(verificationId) {
        return this.start()
            .withUri('/api/user')
            .withParameter('verificationId', verificationId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieve a user_code that is part of an in-progress Device Authorization Grant.
     *
     * This API is useful if you want to build your own login workflow to complete a device grant.
     *
     * @param {string} client_id The client Id.
     * @param {string} client_secret The client Id.
     * @param {string} user_code The end-user verification code.
     * @returns {Promise<ClientResponse<void>>}
     */
    retrieveUserCode(client_id, client_secret, user_code) {
        let body = new url_1.URLSearchParams();
        body.append('client_id', client_id);
        body.append('client_secret', client_secret);
        body.append('user_code', user_code);
        return this.startAnonymous()
            .withUri('/oauth2/device/user-code')
            .withFormData(body)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieve a user_code that is part of an in-progress Device Authorization Grant.
     *
     * This API is useful if you want to build your own login workflow to complete a device grant.
     *
     * This request will require an API key.
     *
     * @param {string} user_code The end-user verification code.
     * @returns {Promise<ClientResponse<void>>}
     */
    retrieveUserCodeUsingAPIKey(user_code) {
        let body = new url_1.URLSearchParams();
        body.append('user_code', user_code);
        return this.startAnonymous()
            .withUri('/oauth2/device/user-code')
            .withFormData(body)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the comments for the user with the given Id.
     *
     * @param {UUID} userId The Id of the user.
     * @returns {Promise<ClientResponse<UserCommentResponse>>}
     */
    retrieveUserComments(userId) {
        return this.start()
            .withUri('/api/user/comment')
            .withUriSegment(userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieve a single User consent by Id.
     *
     * @param {UUID} userConsentId The User consent Id
     * @returns {Promise<ClientResponse<UserConsentResponse>>}
     */
    retrieveUserConsent(userConsentId) {
        return this.start()
            .withUri('/api/user/consent')
            .withUriSegment(userConsentId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the consents for a User.
     *
     * @param {UUID} userId The User's Id
     * @returns {Promise<ClientResponse<UserConsentResponse>>}
     */
    retrieveUserConsents(userId) {
        return this.start()
            .withUri('/api/user/consent')
            .withParameter('userId', userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Call the UserInfo endpoint to retrieve User Claims from the access token issued by FusionAuth.
     *
     * @param {string} encodedJWT The encoded JWT (access token).
     * @returns {Promise<ClientResponse<UserinfoResponse>>}
     */
    retrieveUserInfoFromAccessToken(encodedJWT) {
        return this.startAnonymous()
            .withUri('/oauth2/userinfo')
            .withAuthorization('Bearer ' + encodedJWT)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieve a single Identity Provider user (link).
     *
     * @param {UUID} identityProviderId The unique Id of the identity provider.
     * @param {string} identityProviderUserId The unique Id of the user in the 3rd party identity provider.
     * @param {UUID} userId The unique Id of the FusionAuth user.
     * @returns {Promise<ClientResponse<IdentityProviderLinkResponse>>}
     */
    retrieveUserLink(identityProviderId, identityProviderUserId, userId) {
        return this.start()
            .withUri('/api/identity-provider/link')
            .withParameter('identityProviderId', identityProviderId)
            .withParameter('identityProviderUserId', identityProviderUserId)
            .withParameter('userId', userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieve all Identity Provider users (links) for the user. Specify the optional identityProviderId to retrieve links for a particular IdP.
     *
     * @param {UUID} identityProviderId (Optional) The unique Id of the identity provider. Specify this value to reduce the links returned to those for a particular IdP.
     * @param {UUID} userId The unique Id of the user.
     * @returns {Promise<ClientResponse<IdentityProviderLinkResponse>>}
     */
    retrieveUserLinksByUserId(identityProviderId, userId) {
        return this.start()
            .withUri('/api/identity-provider/link')
            .withParameter('identityProviderId', identityProviderId)
            .withParameter('userId', userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the login report between the two instants for a particular user by Id. If you specify an application Id, it will only return the
     * login counts for that application.
     *
     * @param {UUID} applicationId (Optional) The application Id.
     * @param {UUID} userId The userId Id.
     * @param {number} start The start instant as UTC milliseconds since Epoch.
     * @param {number} end The end instant as UTC milliseconds since Epoch.
     * @returns {Promise<ClientResponse<LoginReportResponse>>}
     */
    retrieveUserLoginReport(applicationId, userId, start, end) {
        return this.start()
            .withUri('/api/report/login')
            .withParameter('applicationId', applicationId)
            .withParameter('userId', userId)
            .withParameter('start', start)
            .withParameter('end', end)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the login report between the two instants for a particular user by login Id. If you specify an application Id, it will only return the
     * login counts for that application.
     *
     * @param {UUID} applicationId (Optional) The application Id.
     * @param {string} loginId The userId Id.
     * @param {number} start The start instant as UTC milliseconds since Epoch.
     * @param {number} end The end instant as UTC milliseconds since Epoch.
     * @returns {Promise<ClientResponse<LoginReportResponse>>}
     */
    retrieveUserLoginReportByLoginId(applicationId, loginId, start, end) {
        return this.start()
            .withUri('/api/report/login')
            .withParameter('applicationId', applicationId)
            .withParameter('loginId', loginId)
            .withParameter('start', start)
            .withParameter('end', end)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the login report between the two instants for a particular user by login Id, using specific loginIdTypes. If you specify an application id, it will only return the
     * login counts for that application.
     *
     * @param {UUID} applicationId (Optional) The application id.
     * @param {string} loginId The userId id.
     * @param {number} start The start instant as UTC milliseconds since Epoch.
     * @param {number} end The end instant as UTC milliseconds since Epoch.
     * @param {Array<String>} loginIdTypes The identity types that FusionAuth will compare the loginId to.
     * @returns {Promise<ClientResponse<LoginReportResponse>>}
     */
    retrieveUserLoginReportByLoginIdAndLoginIdTypes(applicationId, loginId, start, end, loginIdTypes) {
        return this.start()
            .withUri('/api/report/login')
            .withParameter('applicationId', applicationId)
            .withParameter('loginId', loginId)
            .withParameter('start', start)
            .withParameter('end', end)
            .withParameter('loginIdTypes', loginIdTypes)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the last number of login records for a user.
     *
     * @param {UUID} userId The Id of the user.
     * @param {number} offset The initial record. e.g. 0 is the last login, 100 will be the 100th most recent login.
     * @param {number} limit (Optional, defaults to 10) The number of records to retrieve.
     * @returns {Promise<ClientResponse<RecentLoginResponse>>}
     */
    retrieveUserRecentLogins(userId, offset, limit) {
        return this.start()
            .withUri('/api/user/recent-login')
            .withParameter('userId', userId)
            .withParameter('offset', offset)
            .withParameter('limit', limit)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the user for the given Id. This method does not use an API key, instead it uses a JSON Web Token (JWT) for authentication.
     *
     * @param {string} encodedJWT The encoded JWT (access token).
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    retrieveUserUsingJWT(encodedJWT) {
        return this.startAnonymous()
            .withUri('/api/user')
            .withAuthorization('Bearer ' + encodedJWT)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the FusionAuth version string.
     *
     * @returns {Promise<ClientResponse<VersionResponse>>}
     */
    retrieveVersion() {
        return this.start()
            .withUri('/api/system/version')
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the WebAuthn credential for the given Id.
     *
     * @param {UUID} id The Id of the WebAuthn credential.
     * @returns {Promise<ClientResponse<WebAuthnCredentialResponse>>}
     */
    retrieveWebAuthnCredential(id) {
        return this.start()
            .withUri('/api/webauthn')
            .withUriSegment(id)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all WebAuthn credentials for the given user.
     *
     * @param {UUID} userId The user's ID.
     * @returns {Promise<ClientResponse<WebAuthnCredentialResponse>>}
     */
    retrieveWebAuthnCredentialsForUser(userId) {
        return this.start()
            .withUri('/api/webauthn')
            .withParameter('userId', userId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the webhook for the given Id. If you pass in null for the Id, this will return all the webhooks.
     *
     * @param {UUID} webhookId (Optional) The Id of the webhook.
     * @returns {Promise<ClientResponse<WebhookResponse>>}
     */
    retrieveWebhook(webhookId) {
        return this.start()
            .withUri('/api/webhook')
            .withUriSegment(webhookId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves a single webhook attempt log for the given Id.
     *
     * @param {UUID} webhookAttemptLogId The Id of the webhook attempt log to retrieve.
     * @returns {Promise<ClientResponse<WebhookAttemptLogResponse>>}
     */
    retrieveWebhookAttemptLog(webhookAttemptLogId) {
        return this.start()
            .withUri('/api/system/webhook-attempt-log')
            .withUriSegment(webhookAttemptLogId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves a single webhook event log for the given Id.
     *
     * @param {UUID} webhookEventLogId The Id of the webhook event log to retrieve.
     * @returns {Promise<ClientResponse<WebhookEventLogResponse>>}
     */
    retrieveWebhookEventLog(webhookEventLogId) {
        return this.start()
            .withUri('/api/system/webhook-event-log')
            .withUriSegment(webhookEventLogId)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves all the webhooks.
     *
     * @returns {Promise<ClientResponse<WebhookResponse>>}
     */
    retrieveWebhooks() {
        return this.start()
            .withUri('/api/webhook')
            .withMethod("GET")
            .go();
    }
    /**
     * Revokes refresh tokens.
     *
     * Usage examples:
     *   - Delete a single refresh token, pass in only the token.
     *       revokeRefreshToken(token)
     *
     *   - Delete all refresh tokens for a user, pass in only the userId.
     *       revokeRefreshToken(null, userId)
     *
     *   - Delete all refresh tokens for a user for a specific application, pass in both the userId and the applicationId.
     *       revokeRefreshToken(null, userId, applicationId)
     *
     *   - Delete all refresh tokens for an application
     *       revokeRefreshToken(null, null, applicationId)
     *
     * Note: <code>null</code> may be handled differently depending upon the programming language.
     *
     * See also: (method names may vary by language... but you'll figure it out)
     *
     *  - revokeRefreshTokenById
     *  - revokeRefreshTokenByToken
     *  - revokeRefreshTokensByUserId
     *  - revokeRefreshTokensByApplicationId
     *  - revokeRefreshTokensByUserIdForApplication
     *
     * @param {string} token (Optional) The refresh token to delete.
     * @param {UUID} userId (Optional) The user Id whose tokens to delete.
     * @param {UUID} applicationId (Optional) The application Id of the tokens to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    revokeRefreshToken(token, userId, applicationId) {
        return this.start()
            .withUri('/api/jwt/refresh')
            .withParameter('token', token)
            .withParameter('userId', userId)
            .withParameter('applicationId', applicationId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Revokes a single refresh token by the unique Id. The unique Id is not sensitive as it cannot be used to obtain another JWT.
     *
     * @param {UUID} tokenId The unique Id of the token to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    revokeRefreshTokenById(tokenId) {
        return this.start()
            .withUri('/api/jwt/refresh')
            .withUriSegment(tokenId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Revokes a single refresh token by using the actual refresh token value. This refresh token value is sensitive, so  be careful with this API request.
     *
     * @param {string} token The refresh token to delete.
     * @returns {Promise<ClientResponse<void>>}
     */
    revokeRefreshTokenByToken(token) {
        return this.start()
            .withUri('/api/jwt/refresh')
            .withParameter('token', token)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Revoke all refresh tokens that belong to an application by applicationId.
     *
     * @param {UUID} applicationId The unique Id of the application that you want to delete all refresh tokens for.
     * @returns {Promise<ClientResponse<void>>}
     */
    revokeRefreshTokensByApplicationId(applicationId) {
        return this.start()
            .withUri('/api/jwt/refresh')
            .withParameter('applicationId', applicationId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Revoke all refresh tokens that belong to a user by user Id.
     *
     * @param {UUID} userId The unique Id of the user that you want to delete all refresh tokens for.
     * @returns {Promise<ClientResponse<void>>}
     */
    revokeRefreshTokensByUserId(userId) {
        return this.start()
            .withUri('/api/jwt/refresh')
            .withParameter('userId', userId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Revoke all refresh tokens that belong to a user by user Id for a specific application by applicationId.
     *
     * @param {UUID} userId The unique Id of the user that you want to delete all refresh tokens for.
     * @param {UUID} applicationId The unique Id of the application that you want to delete refresh tokens for.
     * @returns {Promise<ClientResponse<void>>}
     */
    revokeRefreshTokensByUserIdForApplication(userId, applicationId) {
        return this.start()
            .withUri('/api/jwt/refresh')
            .withParameter('userId', userId)
            .withParameter('applicationId', applicationId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Revokes refresh tokens using the information in the JSON body. The handling for this method is the same as the revokeRefreshToken method
     * and is based on the information you provide in the RefreshDeleteRequest object. See that method for additional information.
     *
     * @param {RefreshTokenRevokeRequest} request The request information used to revoke the refresh tokens.
     * @returns {Promise<ClientResponse<void>>}
     */
    revokeRefreshTokensWithRequest(request) {
        return this.start()
            .withUri('/api/jwt/refresh')
            .withJSONBody(request)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Revokes a single User consent by Id.
     *
     * @param {UUID} userConsentId The User Consent Id
     * @returns {Promise<ClientResponse<void>>}
     */
    revokeUserConsent(userConsentId) {
        return this.start()
            .withUri('/api/user/consent')
            .withUriSegment(userConsentId)
            .withMethod("DELETE")
            .go();
    }
    /**
     * Searches applications with the specified criteria and pagination.
     *
     * @param {ApplicationSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<ApplicationSearchResponse>>}
     */
    searchApplications(request) {
        return this.start()
            .withUri('/api/application/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches the audit logs with the specified criteria and pagination.
     *
     * @param {AuditLogSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<AuditLogSearchResponse>>}
     */
    searchAuditLogs(request) {
        return this.start()
            .withUri('/api/system/audit-log/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches consents with the specified criteria and pagination.
     *
     * @param {ConsentSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<ConsentSearchResponse>>}
     */
    searchConsents(request) {
        return this.start()
            .withUri('/api/consent/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches email templates with the specified criteria and pagination.
     *
     * @param {EmailTemplateSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<EmailTemplateSearchResponse>>}
     */
    searchEmailTemplates(request) {
        return this.start()
            .withUri('/api/email/template/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches entities with the specified criteria and pagination.
     *
     * @param {EntitySearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<EntitySearchResponse>>}
     */
    searchEntities(request) {
        return this.start()
            .withUri('/api/entity/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Retrieves the entities for the given Ids. If any Id is invalid, it is ignored.
     *
     * @param {Array<string>} ids The entity ids to search for.
     * @returns {Promise<ClientResponse<EntitySearchResponse>>}
     */
    searchEntitiesByIds(ids) {
        return this.start()
            .withUri('/api/entity/search')
            .withParameter('ids', ids)
            .withMethod("GET")
            .go();
    }
    /**
     * Searches Entity Grants with the specified criteria and pagination.
     *
     * @param {EntityGrantSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<EntityGrantSearchResponse>>}
     */
    searchEntityGrants(request) {
        return this.start()
            .withUri('/api/entity/grant/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches the entity types with the specified criteria and pagination.
     *
     * @param {EntityTypeSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<EntityTypeSearchResponse>>}
     */
    searchEntityTypes(request) {
        return this.start()
            .withUri('/api/entity/type/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches the event logs with the specified criteria and pagination.
     *
     * @param {EventLogSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<EventLogSearchResponse>>}
     */
    searchEventLogs(request) {
        return this.start()
            .withUri('/api/system/event-log/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches group members with the specified criteria and pagination.
     *
     * @param {GroupMemberSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<GroupMemberSearchResponse>>}
     */
    searchGroupMembers(request) {
        return this.start()
            .withUri('/api/group/member/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches groups with the specified criteria and pagination.
     *
     * @param {GroupSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<GroupSearchResponse>>}
     */
    searchGroups(request) {
        return this.start()
            .withUri('/api/group/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches the IP Access Control Lists with the specified criteria and pagination.
     *
     * @param {IPAccessControlListSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<IPAccessControlListSearchResponse>>}
     */
    searchIPAccessControlLists(request) {
        return this.start()
            .withUri('/api/ip-acl/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches identity providers with the specified criteria and pagination.
     *
     * @param {IdentityProviderSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<IdentityProviderSearchResponse>>}
     */
    searchIdentityProviders(request) {
        return this.start()
            .withUri('/api/identity-provider/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches keys with the specified criteria and pagination.
     *
     * @param {KeySearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<KeySearchResponse>>}
     */
    searchKeys(request) {
        return this.start()
            .withUri('/api/key/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches lambdas with the specified criteria and pagination.
     *
     * @param {LambdaSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<LambdaSearchResponse>>}
     */
    searchLambdas(request) {
        return this.start()
            .withUri('/api/lambda/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches the login records with the specified criteria and pagination.
     *
     * @param {LoginRecordSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<LoginRecordSearchResponse>>}
     */
    searchLoginRecords(request) {
        return this.start()
            .withUri('/api/system/login-record/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches tenants with the specified criteria and pagination.
     *
     * @param {TenantSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<TenantSearchResponse>>}
     */
    searchTenants(request) {
        return this.start()
            .withUri('/api/tenant/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches themes with the specified criteria and pagination.
     *
     * @param {ThemeSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<ThemeSearchResponse>>}
     */
    searchThemes(request) {
        return this.start()
            .withUri('/api/theme/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches user comments with the specified criteria and pagination.
     *
     * @param {UserCommentSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<UserCommentSearchResponse>>}
     */
    searchUserComments(request) {
        return this.start()
            .withUri('/api/user/comment/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Retrieves the users for the given Ids. If any Id is invalid, it is ignored.
     *
     * @param {Array<string>} ids The user ids to search for.
     * @returns {Promise<ClientResponse<SearchResponse>>}
     *
     * @deprecated This method has been renamed to searchUsersByIds, use that method instead.
     */
    searchUsers(ids) {
        return this.start()
            .withUri('/api/user/search')
            .withParameter('ids', ids)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the users for the given Ids. If any Id is invalid, it is ignored.
     *
     * @param {Array<string>} ids The user Ids to search for.
     * @returns {Promise<ClientResponse<SearchResponse>>}
     */
    searchUsersByIds(ids) {
        return this.start()
            .withUri('/api/user/search')
            .withParameter('ids', ids)
            .withMethod("GET")
            .go();
    }
    /**
     * Retrieves the users for the given search criteria and pagination.
     *
     * @param {SearchRequest} request The search criteria and pagination constraints. Fields used: ids, query, queryString, numberOfResults, orderBy, startRow,
     *    and sortFields.
     * @returns {Promise<ClientResponse<SearchResponse>>}
     */
    searchUsersByQuery(request) {
        return this.start()
            .withUri('/api/user/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Retrieves the users for the given search criteria and pagination.
     *
     * @param {SearchRequest} request The search criteria and pagination constraints. Fields used: ids, query, queryString, numberOfResults, orderBy, startRow,
     *    and sortFields.
     * @returns {Promise<ClientResponse<SearchResponse>>}
     *
     * @deprecated This method has been renamed to searchUsersByQuery, use that method instead.
     */
    searchUsersByQueryString(request) {
        return this.start()
            .withUri('/api/user/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches the webhook event logs with the specified criteria and pagination.
     *
     * @param {WebhookEventLogSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<WebhookEventLogSearchResponse>>}
     */
    searchWebhookEventLogs(request) {
        return this.start()
            .withUri('/api/system/webhook-event-log/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Searches webhooks with the specified criteria and pagination.
     *
     * @param {WebhookSearchRequest} request The search criteria and pagination information.
     * @returns {Promise<ClientResponse<WebhookSearchResponse>>}
     */
    searchWebhooks(request) {
        return this.start()
            .withUri('/api/webhook/search')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Send an email using an email template Id. You can optionally provide <code>requestData</code> to access key value
     * pairs in the email template.
     *
     * @param {UUID} emailTemplateId The Id for the template.
     * @param {SendRequest} request The send email request that contains all the information used to send the email.
     * @returns {Promise<ClientResponse<SendResponse>>}
     */
    sendEmail(emailTemplateId, request) {
        return this.start()
            .withUri('/api/email/send')
            .withUriSegment(emailTemplateId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Sends out an email to a parent that they need to register and create a family or need to log in and add a child to their existing family.
     *
     * @param {FamilyEmailRequest} request The request object that contains the parent email.
     * @returns {Promise<ClientResponse<void>>}
     */
    sendFamilyRequestEmail(request) {
        return this.start()
            .withUri('/api/user/family/request')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Send a passwordless authentication code in an email to complete login.
     *
     * @param {PasswordlessSendRequest} request The passwordless send request that contains all the information used to send an email containing a code.
     * @returns {Promise<ClientResponse<void>>}
     */
    sendPasswordlessCode(request) {
        return this.startAnonymous()
            .withUri('/api/passwordless/send')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Send a Two Factor authentication code to assist in setting up Two Factor authentication or disabling.
     *
     * @param {TwoFactorSendRequest} request The request object that contains all the information used to send the code.
     * @returns {Promise<ClientResponse<void>>}
     *
     * @deprecated This method has been renamed to sendTwoFactorCodeForEnableDisable, use that method instead.
     */
    sendTwoFactorCode(request) {
        return this.start()
            .withUri('/api/two-factor/send')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Send a Two Factor authentication code to assist in setting up Two Factor authentication or disabling.
     *
     * @param {TwoFactorSendRequest} request The request object that contains all the information used to send the code.
     * @returns {Promise<ClientResponse<void>>}
     */
    sendTwoFactorCodeForEnableDisable(request) {
        return this.start()
            .withUri('/api/two-factor/send')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Send a Two Factor authentication code to allow the completion of Two Factor authentication.
     *
     * @param {string} twoFactorId The Id returned by the Login API necessary to complete Two Factor authentication.
     * @returns {Promise<ClientResponse<void>>}
     *
     * @deprecated This method has been renamed to sendTwoFactorCodeForLoginUsingMethod, use that method instead.
     */
    sendTwoFactorCodeForLogin(twoFactorId) {
        return this.startAnonymous()
            .withHeader('Content-Type', 'text/plain')
            .withUri('/api/two-factor/send')
            .withUriSegment(twoFactorId)
            .withMethod("POST")
            .go();
    }
    /**
     * Send a Two Factor authentication code to allow the completion of Two Factor authentication.
     *
     * @param {string} twoFactorId The Id returned by the Login API necessary to complete Two Factor authentication.
     * @param {TwoFactorSendRequest} request The Two Factor send request that contains all the information used to send the Two Factor code to the user.
     * @returns {Promise<ClientResponse<void>>}
     */
    sendTwoFactorCodeForLoginUsingMethod(twoFactorId, request) {
        return this.startAnonymous()
            .withUri('/api/two-factor/send')
            .withUriSegment(twoFactorId)
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Send a verification code using the appropriate transport for the identity type being verified.
     *
     * @param {VerifySendRequest} request The identity verify send request that contains all the information used send the code.
     * @returns {Promise<ClientResponse<void>>}
     */
    sendVerifyIdentity(request) {
        return this.start()
            .withUri('/api/identity/verify/send')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Begins a login request for a 3rd party login that requires user interaction such as HYPR.
     *
     * @param {IdentityProviderStartLoginRequest} request The third-party login request that contains information from the third-party login
     *    providers that FusionAuth uses to reconcile the user's account.
     * @returns {Promise<ClientResponse<IdentityProviderStartLoginResponse>>}
     */
    startIdentityProviderLogin(request) {
        return this.start()
            .withUri('/api/identity-provider/start')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Start a passwordless login request by generating a passwordless code. This code can be sent to the User using the Send
     * Passwordless Code API or using a mechanism outside of FusionAuth. The passwordless login is completed by using the Passwordless Login API with this code.
     *
     * @param {PasswordlessStartRequest} request The passwordless start request that contains all the information used to begin the passwordless login request.
     * @returns {Promise<ClientResponse<PasswordlessStartResponse>>}
     */
    startPasswordlessLogin(request) {
        return this.start()
            .withUri('/api/passwordless/start')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Start a Two-Factor login request by generating a two-factor identifier. This code can then be sent to the Two Factor Send
     * API (/api/two-factor/send)in order to send a one-time use code to a user. You can also use one-time use code returned
     * to send the code out-of-band. The Two-Factor login is completed by making a request to the Two-Factor Login
     * API (/api/two-factor/login). with the two-factor identifier and the one-time use code.
     *
     * This API is intended to allow you to begin a Two-Factor login outside a normal login that originated from the Login API (/api/login).
     *
     * @param {TwoFactorStartRequest} request The Two-Factor start request that contains all the information used to begin the Two-Factor login request.
     * @returns {Promise<ClientResponse<TwoFactorStartResponse>>}
     */
    startTwoFactorLogin(request) {
        return this.start()
            .withUri('/api/two-factor/start')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Start a verification of an identity by generating a code. This code can be sent to the User using the Verify Send API
     * Verification Code API or using a mechanism outside of FusionAuth. The verification is completed by using the Verify Complete API with this code.
     *
     * @param {VerifyStartRequest} request The identity verify start request that contains all the information used to begin the request.
     * @returns {Promise<ClientResponse<VerifyStartResponse>>}
     */
    startVerifyIdentity(request) {
        return this.start()
            .withUri('/api/identity/verify/start')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Start a WebAuthn authentication ceremony by generating a new challenge for the user
     *
     * @param {WebAuthnStartRequest} request An object containing data necessary for starting the authentication ceremony
     * @returns {Promise<ClientResponse<WebAuthnStartResponse>>}
     */
    startWebAuthnLogin(request) {
        return this.start()
            .withUri('/api/webauthn/start')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Start a WebAuthn registration ceremony by generating a new challenge for the user
     *
     * @param {WebAuthnRegisterStartRequest} request An object containing data necessary for starting the registration ceremony
     * @returns {Promise<ClientResponse<WebAuthnRegisterStartResponse>>}
     */
    startWebAuthnRegistration(request) {
        return this.start()
            .withUri('/api/webauthn/register/start')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Complete login using a 2FA challenge
     *
     * @param {TwoFactorLoginRequest} request The login request that contains the user credentials used to log them in.
     * @returns {Promise<ClientResponse<LoginResponse>>}
     */
    twoFactorLogin(request) {
        return this.startAnonymous()
            .withUri('/api/two-factor/login')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Updates an API key with the given Id.
     *
     * @param {UUID} keyId The Id of the API key to update.
     * @param {APIKeyRequest} request The request that contains all the new API key information.
     * @returns {Promise<ClientResponse<APIKeyResponse>>}
     */
    updateAPIKey(keyId, request) {
        return this.start()
            .withUri('/api/api-key')
            .withUriSegment(keyId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the application with the given Id.
     *
     * @param {UUID} applicationId The Id of the application to update.
     * @param {ApplicationRequest} request The request that contains all the new application information.
     * @returns {Promise<ClientResponse<ApplicationResponse>>}
     */
    updateApplication(applicationId, request) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the application role with the given Id for the application.
     *
     * @param {UUID} applicationId The Id of the application that the role belongs to.
     * @param {UUID} roleId The Id of the role to update.
     * @param {ApplicationRequest} request The request that contains all the new role information.
     * @returns {Promise<ClientResponse<ApplicationResponse>>}
     */
    updateApplicationRole(applicationId, roleId, request) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withUriSegment("role")
            .withUriSegment(roleId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the connector with the given Id.
     *
     * @param {UUID} connectorId The Id of the connector to update.
     * @param {ConnectorRequest} request The request object that contains all the new connector information.
     * @returns {Promise<ClientResponse<ConnectorResponse>>}
     */
    updateConnector(connectorId, request) {
        return this.start()
            .withUri('/api/connector')
            .withUriSegment(connectorId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the consent with the given Id.
     *
     * @param {UUID} consentId The Id of the consent to update.
     * @param {ConsentRequest} request The request that contains all the new consent information.
     * @returns {Promise<ClientResponse<ConsentResponse>>}
     */
    updateConsent(consentId, request) {
        return this.start()
            .withUri('/api/consent')
            .withUriSegment(consentId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the email template with the given Id.
     *
     * @param {UUID} emailTemplateId The Id of the email template to update.
     * @param {EmailTemplateRequest} request The request that contains all the new email template information.
     * @returns {Promise<ClientResponse<EmailTemplateResponse>>}
     */
    updateEmailTemplate(emailTemplateId, request) {
        return this.start()
            .withUri('/api/email/template')
            .withUriSegment(emailTemplateId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the Entity with the given Id.
     *
     * @param {UUID} entityId The Id of the Entity to update.
     * @param {EntityRequest} request The request that contains all the new Entity information.
     * @returns {Promise<ClientResponse<EntityResponse>>}
     */
    updateEntity(entityId, request) {
        return this.start()
            .withUri('/api/entity')
            .withUriSegment(entityId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the Entity Type with the given Id.
     *
     * @param {UUID} entityTypeId The Id of the Entity Type to update.
     * @param {EntityTypeRequest} request The request that contains all the new Entity Type information.
     * @returns {Promise<ClientResponse<EntityTypeResponse>>}
     */
    updateEntityType(entityTypeId, request) {
        return this.start()
            .withUri('/api/entity/type')
            .withUriSegment(entityTypeId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the permission with the given Id for the entity type.
     *
     * @param {UUID} entityTypeId The Id of the entityType that the permission belongs to.
     * @param {UUID} permissionId The Id of the permission to update.
     * @param {EntityTypeRequest} request The request that contains all the new permission information.
     * @returns {Promise<ClientResponse<EntityTypeResponse>>}
     */
    updateEntityTypePermission(entityTypeId, permissionId, request) {
        return this.start()
            .withUri('/api/entity/type')
            .withUriSegment(entityTypeId)
            .withUriSegment("permission")
            .withUriSegment(permissionId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates a family with a given Id.
     *
     * @param {UUID} familyId The Id of the family to update.
     * @param {FamilyRequest} request The request object that contains all the new family information.
     * @returns {Promise<ClientResponse<FamilyResponse>>}
     */
    updateFamily(familyId, request) {
        return this.start()
            .withUri('/api/user/family')
            .withUriSegment(familyId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the form with the given Id.
     *
     * @param {UUID} formId The Id of the form to update.
     * @param {FormRequest} request The request object that contains all the new form information.
     * @returns {Promise<ClientResponse<FormResponse>>}
     */
    updateForm(formId, request) {
        return this.start()
            .withUri('/api/form')
            .withUriSegment(formId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the form field with the given Id.
     *
     * @param {UUID} fieldId The Id of the form field to update.
     * @param {FormFieldRequest} request The request object that contains all the new form field information.
     * @returns {Promise<ClientResponse<FormFieldResponse>>}
     */
    updateFormField(fieldId, request) {
        return this.start()
            .withUri('/api/form/field')
            .withUriSegment(fieldId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the group with the given Id.
     *
     * @param {UUID} groupId The Id of the group to update.
     * @param {GroupRequest} request The request that contains all the new group information.
     * @returns {Promise<ClientResponse<GroupResponse>>}
     */
    updateGroup(groupId, request) {
        return this.start()
            .withUri('/api/group')
            .withUriSegment(groupId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Creates a member in a group.
     *
     * @param {MemberRequest} request The request object that contains all the information used to create the group member(s).
     * @returns {Promise<ClientResponse<MemberResponse>>}
     */
    updateGroupMembers(request) {
        return this.start()
            .withUri('/api/group/member')
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the IP Access Control List with the given Id.
     *
     * @param {UUID} accessControlListId The Id of the IP Access Control List to update.
     * @param {IPAccessControlListRequest} request The request that contains all the new IP Access Control List information.
     * @returns {Promise<ClientResponse<IPAccessControlListResponse>>}
     */
    updateIPAccessControlList(accessControlListId, request) {
        return this.start()
            .withUri('/api/ip-acl')
            .withUriSegment(accessControlListId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the identity provider with the given Id.
     *
     * @param {UUID} identityProviderId The Id of the identity provider to update.
     * @param {IdentityProviderRequest} request The request object that contains the updated identity provider.
     * @returns {Promise<ClientResponse<IdentityProviderResponse>>}
     */
    updateIdentityProvider(identityProviderId, request) {
        return this.start()
            .withUri('/api/identity-provider')
            .withUriSegment(identityProviderId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the available integrations.
     *
     * @param {IntegrationRequest} request The request that contains all the new integration information.
     * @returns {Promise<ClientResponse<IntegrationResponse>>}
     */
    updateIntegrations(request) {
        return this.start()
            .withUri('/api/integration')
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the key with the given Id.
     *
     * @param {UUID} keyId The Id of the key to update.
     * @param {KeyRequest} request The request that contains all the new key information.
     * @returns {Promise<ClientResponse<KeyResponse>>}
     */
    updateKey(keyId, request) {
        return this.start()
            .withUri('/api/key')
            .withUriSegment(keyId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the lambda with the given Id.
     *
     * @param {UUID} lambdaId The Id of the lambda to update.
     * @param {LambdaRequest} request The request that contains all the new lambda information.
     * @returns {Promise<ClientResponse<LambdaResponse>>}
     */
    updateLambda(lambdaId, request) {
        return this.start()
            .withUri('/api/lambda')
            .withUriSegment(lambdaId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the message template with the given Id.
     *
     * @param {UUID} messageTemplateId The Id of the message template to update.
     * @param {MessageTemplateRequest} request The request that contains all the new message template information.
     * @returns {Promise<ClientResponse<MessageTemplateResponse>>}
     */
    updateMessageTemplate(messageTemplateId, request) {
        return this.start()
            .withUri('/api/message/template')
            .withUriSegment(messageTemplateId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the messenger with the given Id.
     *
     * @param {UUID} messengerId The Id of the messenger to update.
     * @param {MessengerRequest} request The request object that contains all the new messenger information.
     * @returns {Promise<ClientResponse<MessengerResponse>>}
     */
    updateMessenger(messengerId, request) {
        return this.start()
            .withUri('/api/messenger')
            .withUriSegment(messengerId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the OAuth scope with the given Id for the application.
     *
     * @param {UUID} applicationId The Id of the application that the OAuth scope belongs to.
     * @param {UUID} scopeId The Id of the OAuth scope to update.
     * @param {ApplicationOAuthScopeRequest} request The request that contains all the new OAuth scope information.
     * @returns {Promise<ClientResponse<ApplicationOAuthScopeResponse>>}
     */
    updateOAuthScope(applicationId, scopeId, request) {
        return this.start()
            .withUri('/api/application')
            .withUriSegment(applicationId)
            .withUriSegment("scope")
            .withUriSegment(scopeId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the registration for the user with the given Id and the application defined in the request.
     *
     * @param {UUID} userId The Id of the user whose registration is going to be updated.
     * @param {RegistrationRequest} request The request that contains all the new registration information.
     * @returns {Promise<ClientResponse<RegistrationResponse>>}
     */
    updateRegistration(userId, request) {
        return this.start()
            .withUri('/api/user/registration')
            .withUriSegment(userId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the system configuration.
     *
     * @param {SystemConfigurationRequest} request The request that contains all the new system configuration information.
     * @returns {Promise<ClientResponse<SystemConfigurationResponse>>}
     */
    updateSystemConfiguration(request) {
        return this.start()
            .withUri('/api/system-configuration')
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the tenant with the given Id.
     *
     * @param {UUID} tenantId The Id of the tenant to update.
     * @param {TenantRequest} request The request that contains all the new tenant information.
     * @returns {Promise<ClientResponse<TenantResponse>>}
     */
    updateTenant(tenantId, request) {
        return this.start()
            .withUri('/api/tenant')
            .withUriSegment(tenantId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the theme with the given Id.
     *
     * @param {UUID} themeId The Id of the theme to update.
     * @param {ThemeRequest} request The request that contains all the new theme information.
     * @returns {Promise<ClientResponse<ThemeResponse>>}
     */
    updateTheme(themeId, request) {
        return this.start()
            .withUri('/api/theme')
            .withUriSegment(themeId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the user with the given Id.
     *
     * @param {UUID} userId The Id of the user to update.
     * @param {UserRequest} request The request that contains all the new user information.
     * @returns {Promise<ClientResponse<UserResponse>>}
     */
    updateUser(userId, request) {
        return this.start()
            .withUri('/api/user')
            .withUriSegment(userId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the user action with the given Id.
     *
     * @param {UUID} userActionId The Id of the user action to update.
     * @param {UserActionRequest} request The request that contains all the new user action information.
     * @returns {Promise<ClientResponse<UserActionResponse>>}
     */
    updateUserAction(userActionId, request) {
        return this.start()
            .withUri('/api/user-action')
            .withUriSegment(userActionId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the user action reason with the given Id.
     *
     * @param {UUID} userActionReasonId The Id of the user action reason to update.
     * @param {UserActionReasonRequest} request The request that contains all the new user action reason information.
     * @returns {Promise<ClientResponse<UserActionReasonResponse>>}
     */
    updateUserActionReason(userActionReasonId, request) {
        return this.start()
            .withUri('/api/user-action-reason')
            .withUriSegment(userActionReasonId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates a single User consent by Id.
     *
     * @param {UUID} userConsentId The User Consent Id
     * @param {UserConsentRequest} request The request that contains the user consent information.
     * @returns {Promise<ClientResponse<UserConsentResponse>>}
     */
    updateUserConsent(userConsentId, request) {
        return this.start()
            .withUri('/api/user/consent')
            .withUriSegment(userConsentId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Updates the webhook with the given Id.
     *
     * @param {UUID} webhookId The Id of the webhook to update.
     * @param {WebhookRequest} request The request that contains all the new webhook information.
     * @returns {Promise<ClientResponse<WebhookResponse>>}
     */
    updateWebhook(webhookId, request) {
        return this.start()
            .withUri('/api/webhook')
            .withUriSegment(webhookId)
            .withJSONBody(request)
            .withMethod("PUT")
            .go();
    }
    /**
     * Creates or updates an Entity Grant. This is when a User/Entity is granted permissions to an Entity.
     *
     * @param {UUID} entityId The Id of the Entity that the User/Entity is being granted access to.
     * @param {EntityGrantRequest} request The request object that contains all the information used to create the Entity Grant.
     * @returns {Promise<ClientResponse<void>>}
     */
    upsertEntityGrant(entityId, request) {
        return this.start()
            .withUri('/api/entity')
            .withUriSegment(entityId)
            .withUriSegment("grant")
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Validates the end-user provided user_code from the user-interaction of the Device Authorization Grant.
     * If you build your own activation form you should validate the user provided code prior to beginning the Authorization grant.
     *
     * @param {string} user_code The end-user verification code.
     * @param {string} client_id The client Id.
     * @returns {Promise<ClientResponse<void>>}
     */
    validateDevice(user_code, client_id) {
        return this.startAnonymous()
            .withUri('/oauth2/device/validate')
            .withParameter('user_code', user_code)
            .withParameter('client_id', client_id)
            .withMethod("GET")
            .go();
    }
    /**
     * Validates the provided JWT (encoded JWT string) to ensure the token is valid. A valid access token is properly
     * signed and not expired.
     * <p>
     * This API may be used to verify the JWT as well as decode the encoded JWT into human readable identity claims.
     *
     * @param {string} encodedJWT The encoded JWT (access token).
     * @returns {Promise<ClientResponse<ValidateResponse>>}
     */
    validateJWT(encodedJWT) {
        return this.startAnonymous()
            .withUri('/api/jwt/validate')
            .withAuthorization('Bearer ' + encodedJWT)
            .withMethod("GET")
            .go();
    }
    /**
     * It's a JWT vending machine!
     *
     * Issue a new access token (JWT) with the provided claims in the request. This JWT is not scoped to a tenant or user, it is a free form
     * token that will contain what claims you provide.
     * <p>
     * The iat, exp and jti claims will be added by FusionAuth, all other claims must be provided by the caller.
     *
     * If a TTL is not provided in the request, the TTL will be retrieved from the default Tenant or the Tenant specified on the request either
     * by way of the X-FusionAuth-TenantId request header, or a tenant scoped API key.
     *
     * @param {JWTVendRequest} request The request that contains all the claims for this JWT.
     * @returns {Promise<ClientResponse<JWTVendResponse>>}
     */
    vendJWT(request) {
        return this.start()
            .withUri('/api/jwt/vend')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Confirms a email verification. The Id given is usually from an email sent to the user.
     *
     * @param {string} verificationId The email verification Id sent to the user.
     * @returns {Promise<ClientResponse<void>>}
     *
     * @deprecated This method has been renamed to verifyEmailAddress and changed to take a JSON request body, use that method instead.
     */
    verifyEmail(verificationId) {
        return this.startAnonymous()
            .withHeader('Content-Type', 'text/plain')
            .withUri('/api/user/verify-email')
            .withUriSegment(verificationId)
            .withMethod("POST")
            .go();
    }
    /**
     * Confirms a user's email address.
     *
     * The request body will contain the verificationId. You may also be required to send a one-time use code based upon your configuration. When
     * the tenant is configured to gate a user until their email address is verified, this procedures requires two values instead of one.
     * The verificationId is a high entropy value and the one-time use code is a low entropy value that is easily entered in a user interactive form. The
     * two values together are able to confirm a user's email address and mark the user's email address as verified.
     *
     * @param {VerifyEmailRequest} request The request that contains the verificationId and optional one-time use code paired with the verificationId.
     * @returns {Promise<ClientResponse<void>>}
     */
    verifyEmailAddress(request) {
        return this.startAnonymous()
            .withUri('/api/user/verify-email')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Administratively verify a user's email address. Use this method to bypass email verification for the user.
     *
     * The request body will contain the userId to be verified. An API key is required when sending the userId in the request body.
     *
     * @param {VerifyEmailRequest} request The request that contains the userId to verify.
     * @returns {Promise<ClientResponse<void>>}
     */
    verifyEmailAddressByUserId(request) {
        return this.start()
            .withUri('/api/user/verify-email')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Administratively verify a user identity.
     *
     * @param {VerifyRequest} request The identity verify request that contains information to verify the identity.
     * @returns {Promise<ClientResponse<void>>}
     */
    verifyIdentity(request) {
        return this.start()
            .withUri('/api/identity/verify')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /**
     * Confirms an application registration. The Id given is usually from an email sent to the user.
     *
     * @param {string} verificationId The registration verification Id sent to the user.
     * @returns {Promise<ClientResponse<void>>}
     *
     * @deprecated This method has been renamed to verifyUserRegistration and changed to take a JSON request body, use that method instead.
     */
    verifyRegistration(verificationId) {
        return this.startAnonymous()
            .withHeader('Content-Type', 'text/plain')
            .withUri('/api/user/verify-registration')
            .withUriSegment(verificationId)
            .withMethod("POST")
            .go();
    }
    /**
     * Confirms a user's registration.
     *
     * The request body will contain the verificationId. You may also be required to send a one-time use code based upon your configuration. When
     * the application is configured to gate a user until their registration is verified, this procedures requires two values instead of one.
     * The verificationId is a high entropy value and the one-time use code is a low entropy value that is easily entered in a user interactive form. The
     * two values together are able to confirm a user's registration and mark the user's registration as verified.
     *
     * @param {VerifyRegistrationRequest} request The request that contains the verificationId and optional one-time use code paired with the verificationId.
     * @returns {Promise<ClientResponse<void>>}
     */
    verifyUserRegistration(request) {
        return this.startAnonymous()
            .withUri('/api/user/verify-registration')
            .withJSONBody(request)
            .withMethod("POST")
            .go();
    }
    /* ===================================================================================================================
     * Private methods
     * ===================================================================================================================*/
    /**
     * creates a rest client
     *
     * @returns The RESTClient that will be used to call.
     * @private
     */
    start() {
        return this.startAnonymous()
            .withAuthorization(this.apiKey);
    }
    startAnonymous() {
        let client = this.clientBuilder.build(this.host);
        if (this.tenantId != null) {
            client.withHeader('X-FusionAuth-TenantId', this.tenantId);
        }
        if (this.credentials != null) {
            client.withCredentials(this.credentials);
        }
        return client;
    }
}
exports.FusionAuthClient = FusionAuthClient;
exports.default = FusionAuthClient;
/**
 * Available JSON Web Algorithms (JWA) as described in RFC 7518 available for this JWT implementation.
 *
 * @author Daniel DeGroff
 */
var Algorithm;
(function (Algorithm) {
    Algorithm["ES256"] = "ES256";
    Algorithm["ES384"] = "ES384";
    Algorithm["ES512"] = "ES512";
    Algorithm["HS256"] = "HS256";
    Algorithm["HS384"] = "HS384";
    Algorithm["HS512"] = "HS512";
    Algorithm["PS256"] = "PS256";
    Algorithm["PS384"] = "PS384";
    Algorithm["PS512"] = "PS512";
    Algorithm["RS256"] = "RS256";
    Algorithm["RS384"] = "RS384";
    Algorithm["RS512"] = "RS512";
    Algorithm["none"] = "none";
})(Algorithm = exports.Algorithm || (exports.Algorithm = {}));
//      This is separate from IdentityType.
var LoginIdType;
(function (LoginIdType) {
    LoginIdType["email"] = "email";
    LoginIdType["phoneNumber"] = "phoneNumber";
    LoginIdType["username"] = "username";
})(LoginIdType = exports.LoginIdType || (exports.LoginIdType = {}));
var RegistrationType;
(function (RegistrationType) {
    RegistrationType["basic"] = "basic";
    RegistrationType["advanced"] = "advanced";
})(RegistrationType = exports.RegistrationType || (exports.RegistrationType = {}));
var SAMLLogoutBehavior;
(function (SAMLLogoutBehavior) {
    SAMLLogoutBehavior["AllParticipants"] = "AllParticipants";
    SAMLLogoutBehavior["OnlyOriginator"] = "OnlyOriginator";
})(SAMLLogoutBehavior = exports.SAMLLogoutBehavior || (exports.SAMLLogoutBehavior = {}));
var XMLSignatureLocation;
(function (XMLSignatureLocation) {
    XMLSignatureLocation["Assertion"] = "Assertion";
    XMLSignatureLocation["Response"] = "Response";
})(XMLSignatureLocation = exports.XMLSignatureLocation || (exports.XMLSignatureLocation = {}));
/**
 * @author Daniel DeGroff
 */
var ApplicationMultiFactorTrustPolicy;
(function (ApplicationMultiFactorTrustPolicy) {
    ApplicationMultiFactorTrustPolicy["Any"] = "Any";
    ApplicationMultiFactorTrustPolicy["This"] = "This";
    ApplicationMultiFactorTrustPolicy["None"] = "None";
})(ApplicationMultiFactorTrustPolicy = exports.ApplicationMultiFactorTrustPolicy || (exports.ApplicationMultiFactorTrustPolicy = {}));
/**
 * Used to communicate whether and how authenticator attestation should be delivered to the Relying Party
 *
 * @author Spencer Witt
 */
var AttestationConveyancePreference;
(function (AttestationConveyancePreference) {
    AttestationConveyancePreference["none"] = "none";
    AttestationConveyancePreference["indirect"] = "indirect";
    AttestationConveyancePreference["direct"] = "direct";
    AttestationConveyancePreference["enterprise"] = "enterprise";
})(AttestationConveyancePreference = exports.AttestationConveyancePreference || (exports.AttestationConveyancePreference = {}));
/**
 * Used to indicate what type of attestation was included in the authenticator response for a given WebAuthn credential at the time it was created
 *
 * @author Spencer Witt
 */
var AttestationType;
(function (AttestationType) {
    AttestationType["basic"] = "basic";
    AttestationType["self"] = "self";
    AttestationType["attestationCa"] = "attestationCa";
    AttestationType["anonymizationCa"] = "anonymizationCa";
    AttestationType["none"] = "none";
})(AttestationType = exports.AttestationType || (exports.AttestationType = {}));
/**
 * @author Brett Pontarelli
 */
var AuthenticationThreats;
(function (AuthenticationThreats) {
    AuthenticationThreats["ImpossibleTravel"] = "ImpossibleTravel";
})(AuthenticationThreats = exports.AuthenticationThreats || (exports.AuthenticationThreats = {}));
/**
 * Describes the <a href="https://www.w3.org/TR/webauthn-2/#authenticator-attachment-modality">authenticator attachment modality</a>.
 *
 * @author Spencer Witt
 */
var AuthenticatorAttachment;
(function (AuthenticatorAttachment) {
    AuthenticatorAttachment["platform"] = "platform";
    AuthenticatorAttachment["crossPlatform"] = "crossPlatform";
})(AuthenticatorAttachment = exports.AuthenticatorAttachment || (exports.AuthenticatorAttachment = {}));
/**
 * Describes the authenticator attachment modality preference for a WebAuthn workflow. See {@link AuthenticatorAttachment}
 *
 * @author Spencer Witt
 */
var AuthenticatorAttachmentPreference;
(function (AuthenticatorAttachmentPreference) {
    AuthenticatorAttachmentPreference["any"] = "any";
    AuthenticatorAttachmentPreference["platform"] = "platform";
    AuthenticatorAttachmentPreference["crossPlatform"] = "crossPlatform";
})(AuthenticatorAttachmentPreference = exports.AuthenticatorAttachmentPreference || (exports.AuthenticatorAttachmentPreference = {}));
var TOTPAlgorithm;
(function (TOTPAlgorithm) {
    TOTPAlgorithm["HmacSHA1"] = "HmacSHA1";
    TOTPAlgorithm["HmacSHA256"] = "HmacSHA256";
    TOTPAlgorithm["HmacSHA512"] = "HmacSHA512";
})(TOTPAlgorithm = exports.TOTPAlgorithm || (exports.TOTPAlgorithm = {}));
/**
 * @author Daniel DeGroff
 */
var BreachedPasswordStatus;
(function (BreachedPasswordStatus) {
    BreachedPasswordStatus["None"] = "None";
    BreachedPasswordStatus["ExactMatch"] = "ExactMatch";
    BreachedPasswordStatus["SubAddressMatch"] = "SubAddressMatch";
    BreachedPasswordStatus["PasswordOnly"] = "PasswordOnly";
    BreachedPasswordStatus["CommonPassword"] = "CommonPassword";
})(BreachedPasswordStatus = exports.BreachedPasswordStatus || (exports.BreachedPasswordStatus = {}));
/**
 * XML canonicalization method enumeration. This is used for the IdP and SP side of FusionAuth SAML.
 *
 * @author Brian Pontarelli
 */
var CanonicalizationMethod;
(function (CanonicalizationMethod) {
    CanonicalizationMethod["exclusive"] = "exclusive";
    CanonicalizationMethod["exclusive_with_comments"] = "exclusive_with_comments";
    CanonicalizationMethod["inclusive"] = "inclusive";
    CanonicalizationMethod["inclusive_with_comments"] = "inclusive_with_comments";
})(CanonicalizationMethod = exports.CanonicalizationMethod || (exports.CanonicalizationMethod = {}));
/**
 * @author Brett Pontarelli
 */
var CaptchaMethod;
(function (CaptchaMethod) {
    CaptchaMethod["GoogleRecaptchaV2"] = "GoogleRecaptchaV2";
    CaptchaMethod["GoogleRecaptchaV3"] = "GoogleRecaptchaV3";
    CaptchaMethod["HCaptcha"] = "HCaptcha";
    CaptchaMethod["HCaptchaEnterprise"] = "HCaptchaEnterprise";
})(CaptchaMethod = exports.CaptchaMethod || (exports.CaptchaMethod = {}));
/**
 * @author Trevor Smith
 */
var ChangePasswordReason;
(function (ChangePasswordReason) {
    ChangePasswordReason["Administrative"] = "Administrative";
    ChangePasswordReason["Breached"] = "Breached";
    ChangePasswordReason["Expired"] = "Expired";
    ChangePasswordReason["Validation"] = "Validation";
})(ChangePasswordReason = exports.ChangePasswordReason || (exports.ChangePasswordReason = {}));
/**
 * @author Brett Guy
 */
var ClientAuthenticationPolicy;
(function (ClientAuthenticationPolicy) {
    ClientAuthenticationPolicy["Required"] = "Required";
    ClientAuthenticationPolicy["NotRequired"] = "NotRequired";
    ClientAuthenticationPolicy["NotRequiredWhenUsingPKCE"] = "NotRequiredWhenUsingPKCE";
})(ClientAuthenticationPolicy = exports.ClientAuthenticationPolicy || (exports.ClientAuthenticationPolicy = {}));
/**
 * The types of connectors. This enum is stored as an ordinal on the <code>identities</code> table, order must be maintained.
 *
 * @author Trevor Smith
 */
var ConnectorType;
(function (ConnectorType) {
    ConnectorType["FusionAuth"] = "FusionAuth";
    ConnectorType["Generic"] = "Generic";
    ConnectorType["LDAP"] = "LDAP";
})(ConnectorType = exports.ConnectorType || (exports.ConnectorType = {}));
/**
 * Models a consent.
 *
 * @author Daniel DeGroff
 */
var ConsentStatus;
(function (ConsentStatus) {
    ConsentStatus["Active"] = "Active";
    ConsentStatus["Revoked"] = "Revoked";
})(ConsentStatus = exports.ConsentStatus || (exports.ConsentStatus = {}));
/**
 * Status for content like usernames, profile attributes, etc.
 *
 * @author Brian Pontarelli
 */
var ContentStatus;
(function (ContentStatus) {
    ContentStatus["ACTIVE"] = "ACTIVE";
    ContentStatus["PENDING"] = "PENDING";
    ContentStatus["REJECTED"] = "REJECTED";
})(ContentStatus = exports.ContentStatus || (exports.ContentStatus = {}));
/**
 * A number identifying a cryptographic algorithm. Values should be registered with the <a
 * href="https://www.iana.org/assignments/cose/cose.xhtml#algorithms">IANA COSE Algorithms registry</a>
 *
 * @author Spencer Witt
 */
var CoseAlgorithmIdentifier;
(function (CoseAlgorithmIdentifier) {
    CoseAlgorithmIdentifier["ES256"] = "SHA256withECDSA";
    CoseAlgorithmIdentifier["ES384"] = "SHA384withECDSA";
    CoseAlgorithmIdentifier["ES512"] = "SHA512withECDSA";
    CoseAlgorithmIdentifier["RS256"] = "SHA256withRSA";
    CoseAlgorithmIdentifier["RS384"] = "SHA384withRSA";
    CoseAlgorithmIdentifier["RS512"] = "SHA512withRSA";
    CoseAlgorithmIdentifier["PS256"] = "SHA-256";
    CoseAlgorithmIdentifier["PS384"] = "SHA-384";
    CoseAlgorithmIdentifier["PS512"] = "SHA-512";
})(CoseAlgorithmIdentifier = exports.CoseAlgorithmIdentifier || (exports.CoseAlgorithmIdentifier = {}));
/**
 * COSE Elliptic Curve identifier to determine which elliptic curve to use with a given key
 *
 * @author Spencer Witt
 */
var CoseEllipticCurve;
(function (CoseEllipticCurve) {
    CoseEllipticCurve["Reserved"] = "Reserved";
    CoseEllipticCurve["P256"] = "P256";
    CoseEllipticCurve["P384"] = "P384";
    CoseEllipticCurve["P521"] = "P521";
    CoseEllipticCurve["X25519"] = "X25519";
    CoseEllipticCurve["X448"] = "X448";
    CoseEllipticCurve["Ed25519"] = "Ed25519";
    CoseEllipticCurve["Ed448"] = "Ed448";
    CoseEllipticCurve["Secp256k1"] = "Secp256k1";
})(CoseEllipticCurve = exports.CoseEllipticCurve || (exports.CoseEllipticCurve = {}));
/**
 * COSE key type
 *
 * @author Spencer Witt
 */
var CoseKeyType;
(function (CoseKeyType) {
    CoseKeyType["Reserved"] = "0";
    CoseKeyType["OKP"] = "1";
    CoseKeyType["EC2"] = "2";
    CoseKeyType["RSA"] = "3";
    CoseKeyType["Symmetric"] = "4";
})(CoseKeyType = exports.CoseKeyType || (exports.CoseKeyType = {}));
var DeviceType;
(function (DeviceType) {
    DeviceType["BROWSER"] = "BROWSER";
    DeviceType["DESKTOP"] = "DESKTOP";
    DeviceType["LAPTOP"] = "LAPTOP";
    DeviceType["MOBILE"] = "MOBILE";
    DeviceType["OTHER"] = "OTHER";
    DeviceType["SERVER"] = "SERVER";
    DeviceType["TABLET"] = "TABLET";
    DeviceType["TV"] = "TV";
    DeviceType["UNKNOWN"] = "UNKNOWN";
})(DeviceType = exports.DeviceType || (exports.DeviceType = {}));
var EmailSecurityType;
(function (EmailSecurityType) {
    EmailSecurityType["NONE"] = "NONE";
    EmailSecurityType["SSL"] = "SSL";
    EmailSecurityType["TLS"] = "TLS";
})(EmailSecurityType = exports.EmailSecurityType || (exports.EmailSecurityType = {}));
/**
 * Event Log Type
 *
 * @author Daniel DeGroff
 */
var EventLogType;
(function (EventLogType) {
    EventLogType["Information"] = "Information";
    EventLogType["Debug"] = "Debug";
    EventLogType["Error"] = "Error";
})(EventLogType = exports.EventLogType || (exports.EventLogType = {}));
/**
 * Models the event types that FusionAuth produces.
 *
 * @author Brian Pontarelli
 */
var EventType;
(function (EventType) {
    EventType["JWTPublicKeyUpdate"] = "jwt.public-key.update";
    EventType["JWTRefreshTokenRevoke"] = "jwt.refresh-token.revoke";
    EventType["JWTRefresh"] = "jwt.refresh";
    EventType["AuditLogCreate"] = "audit-log.create";
    EventType["EventLogCreate"] = "event-log.create";
    EventType["KickstartSuccess"] = "kickstart.success";
    EventType["GroupCreate"] = "group.create";
    EventType["GroupCreateComplete"] = "group.create.complete";
    EventType["GroupDelete"] = "group.delete";
    EventType["GroupDeleteComplete"] = "group.delete.complete";
    EventType["GroupMemberAdd"] = "group.member.add";
    EventType["GroupMemberAddComplete"] = "group.member.add.complete";
    EventType["GroupMemberRemove"] = "group.member.remove";
    EventType["GroupMemberRemoveComplete"] = "group.member.remove.complete";
    EventType["GroupMemberUpdate"] = "group.member.update";
    EventType["GroupMemberUpdateComplete"] = "group.member.update.complete";
    EventType["GroupUpdate"] = "group.update";
    EventType["GroupUpdateComplete"] = "group.update.complete";
    EventType["UserAction"] = "user.action";
    EventType["UserBulkCreate"] = "user.bulk.create";
    EventType["UserCreate"] = "user.create";
    EventType["UserCreateComplete"] = "user.create.complete";
    EventType["UserDeactivate"] = "user.deactivate";
    EventType["UserDelete"] = "user.delete";
    EventType["UserDeleteComplete"] = "user.delete.complete";
    EventType["UserEmailUpdate"] = "user.email.update";
    EventType["UserEmailVerified"] = "user.email.verified";
    EventType["UserIdentityProviderLink"] = "user.identity-provider.link";
    EventType["UserIdentityProviderUnlink"] = "user.identity-provider.unlink";
    EventType["UserLoginIdDuplicateOnCreate"] = "user.loginId.duplicate.create";
    EventType["UserLoginIdDuplicateOnUpdate"] = "user.loginId.duplicate.update";
    EventType["UserLoginFailed"] = "user.login.failed";
    EventType["UserLoginNewDevice"] = "user.login.new-device";
    EventType["UserLoginSuccess"] = "user.login.success";
    EventType["UserLoginSuspicious"] = "user.login.suspicious";
    EventType["UserPasswordBreach"] = "user.password.breach";
    EventType["UserPasswordResetSend"] = "user.password.reset.send";
    EventType["UserPasswordResetStart"] = "user.password.reset.start";
    EventType["UserPasswordResetSuccess"] = "user.password.reset.success";
    EventType["UserPasswordUpdate"] = "user.password.update";
    EventType["UserReactivate"] = "user.reactivate";
    EventType["UserRegistrationCreate"] = "user.registration.create";
    EventType["UserRegistrationCreateComplete"] = "user.registration.create.complete";
    EventType["UserRegistrationDelete"] = "user.registration.delete";
    EventType["UserRegistrationDeleteComplete"] = "user.registration.delete.complete";
    EventType["UserRegistrationUpdate"] = "user.registration.update";
    EventType["UserRegistrationUpdateComplete"] = "user.registration.update.complete";
    EventType["UserRegistrationVerified"] = "user.registration.verified";
    EventType["UserTwoFactorMethodAdd"] = "user.two-factor.method.add";
    EventType["UserTwoFactorMethodRemove"] = "user.two-factor.method.remove";
    EventType["UserUpdate"] = "user.update";
    EventType["UserUpdateComplete"] = "user.update.complete";
    EventType["Test"] = "test";
    EventType["UserIdentityVerified"] = "user.identity.verified";
    EventType["UserIdentityUpdate"] = "user.identity.update";
})(EventType = exports.EventType || (exports.EventType = {}));
/**
 * @author Brian Pontarelli
 */
var ExpiryUnit;
(function (ExpiryUnit) {
    ExpiryUnit["MINUTES"] = "MINUTES";
    ExpiryUnit["HOURS"] = "HOURS";
    ExpiryUnit["DAYS"] = "DAYS";
    ExpiryUnit["WEEKS"] = "WEEKS";
    ExpiryUnit["MONTHS"] = "MONTHS";
    ExpiryUnit["YEARS"] = "YEARS";
})(ExpiryUnit = exports.ExpiryUnit || (exports.ExpiryUnit = {}));
var FamilyRole;
(function (FamilyRole) {
    FamilyRole["Child"] = "Child";
    FamilyRole["Teen"] = "Teen";
    FamilyRole["Adult"] = "Adult";
})(FamilyRole = exports.FamilyRole || (exports.FamilyRole = {}));
/**
 * @author Daniel DeGroff
 */
var FormControl;
(function (FormControl) {
    FormControl["checkbox"] = "checkbox";
    FormControl["number"] = "number";
    FormControl["password"] = "password";
    FormControl["radio"] = "radio";
    FormControl["select"] = "select";
    FormControl["textarea"] = "textarea";
    FormControl["text"] = "text";
})(FormControl = exports.FormControl || (exports.FormControl = {}));
/**
 * @author Daniel DeGroff
 */
var FormDataType;
(function (FormDataType) {
    FormDataType["bool"] = "bool";
    FormDataType["consent"] = "consent";
    FormDataType["date"] = "date";
    FormDataType["email"] = "email";
    FormDataType["number"] = "number";
    FormDataType["phoneNumber"] = "phoneNumber";
    FormDataType["string"] = "string";
})(FormDataType = exports.FormDataType || (exports.FormDataType = {}));
/**
 * @author Daniel DeGroff
 */
var FormFieldAdminPolicy;
(function (FormFieldAdminPolicy) {
    FormFieldAdminPolicy["Edit"] = "Edit";
    FormFieldAdminPolicy["View"] = "View";
})(FormFieldAdminPolicy = exports.FormFieldAdminPolicy || (exports.FormFieldAdminPolicy = {}));
/**
 * @author Daniel DeGroff
 */
var FormType;
(function (FormType) {
    FormType["registration"] = "registration";
    FormType["adminRegistration"] = "adminRegistration";
    FormType["adminUser"] = "adminUser";
    FormType["selfServiceUser"] = "selfServiceUser";
})(FormType = exports.FormType || (exports.FormType = {}));
/**
 * Authorization Grant types as defined by the <a href="https://tools.ietf.org/html/rfc6749">OAuth 2.0 Authorization
 * Framework - RFC 6749</a>.
 * <p>
 * Specific names as defined by <a href="https://tools.ietf.org/html/rfc7591#section-4.1">
 * OAuth 2.0 Dynamic Client Registration Protocol - RFC 7591 Section 4.1</a>
 *
 * @author Daniel DeGroff
 */
var GrantType;
(function (GrantType) {
    GrantType["authorization_code"] = "authorization_code";
    GrantType["implicit"] = "implicit";
    GrantType["password"] = "password";
    GrantType["client_credentials"] = "client_credentials";
    GrantType["refresh_token"] = "refresh_token";
    GrantType["unknown"] = "unknown";
    GrantType["device_code"] = "urn:ietf:params:oauth:grant-type:device_code";
})(GrantType = exports.GrantType || (exports.GrantType = {}));
/**
 * @author Daniel DeGroff
 */
var HTTPMethod;
(function (HTTPMethod) {
    HTTPMethod["GET"] = "GET";
    HTTPMethod["POST"] = "POST";
    HTTPMethod["PUT"] = "PUT";
    HTTPMethod["DELETE"] = "DELETE";
    HTTPMethod["HEAD"] = "HEAD";
    HTTPMethod["OPTIONS"] = "OPTIONS";
    HTTPMethod["PATCH"] = "PATCH";
})(HTTPMethod = exports.HTTPMethod || (exports.HTTPMethod = {}));
/**
 * @author Brett Guy
 */
var IPAccessControlEntryAction;
(function (IPAccessControlEntryAction) {
    IPAccessControlEntryAction["Allow"] = "Allow";
    IPAccessControlEntryAction["Block"] = "Block";
})(IPAccessControlEntryAction = exports.IPAccessControlEntryAction || (exports.IPAccessControlEntryAction = {}));
/**
 * The IdP behavior when no user link has been made yet.
 *
 * @author Daniel DeGroff
 */
var IdentityProviderLinkingStrategy;
(function (IdentityProviderLinkingStrategy) {
    IdentityProviderLinkingStrategy["CreatePendingLink"] = "CreatePendingLink";
    IdentityProviderLinkingStrategy["Disabled"] = "Disabled";
    IdentityProviderLinkingStrategy["LinkAnonymously"] = "LinkAnonymously";
    IdentityProviderLinkingStrategy["LinkByEmail"] = "LinkByEmail";
    IdentityProviderLinkingStrategy["LinkByEmailForExistingUser"] = "LinkByEmailForExistingUser";
    IdentityProviderLinkingStrategy["LinkByUsername"] = "LinkByUsername";
    IdentityProviderLinkingStrategy["LinkByUsernameForExistingUser"] = "LinkByUsernameForExistingUser";
    IdentityProviderLinkingStrategy["Unsupported"] = "Unsupported";
})(IdentityProviderLinkingStrategy = exports.IdentityProviderLinkingStrategy || (exports.IdentityProviderLinkingStrategy = {}));
/**
 * @author Brett Pontarelli
 */
var IdentityProviderLoginMethod;
(function (IdentityProviderLoginMethod) {
    IdentityProviderLoginMethod["UsePopup"] = "UsePopup";
    IdentityProviderLoginMethod["UseRedirect"] = "UseRedirect";
    IdentityProviderLoginMethod["UseVendorJavaScript"] = "UseVendorJavaScript";
})(IdentityProviderLoginMethod = exports.IdentityProviderLoginMethod || (exports.IdentityProviderLoginMethod = {}));
var ClientAuthenticationMethod;
(function (ClientAuthenticationMethod) {
    ClientAuthenticationMethod["none"] = "none";
    ClientAuthenticationMethod["client_secret_basic"] = "client_secret_basic";
    ClientAuthenticationMethod["client_secret_post"] = "client_secret_post";
})(ClientAuthenticationMethod = exports.ClientAuthenticationMethod || (exports.ClientAuthenticationMethod = {}));
/**
 * @author Daniel DeGroff
 */
var IdentityProviderType;
(function (IdentityProviderType) {
    IdentityProviderType["Apple"] = "Apple";
    IdentityProviderType["EpicGames"] = "EpicGames";
    IdentityProviderType["ExternalJWT"] = "ExternalJWT";
    IdentityProviderType["Facebook"] = "Facebook";
    IdentityProviderType["Google"] = "Google";
    IdentityProviderType["HYPR"] = "HYPR";
    IdentityProviderType["LinkedIn"] = "LinkedIn";
    IdentityProviderType["Nintendo"] = "Nintendo";
    IdentityProviderType["OpenIDConnect"] = "OpenIDConnect";
    IdentityProviderType["SAMLv2"] = "SAMLv2";
    IdentityProviderType["SAMLv2IdPInitiated"] = "SAMLv2IdPInitiated";
    IdentityProviderType["SonyPSN"] = "SonyPSN";
    IdentityProviderType["Steam"] = "Steam";
    IdentityProviderType["Twitch"] = "Twitch";
    IdentityProviderType["Twitter"] = "Twitter";
    IdentityProviderType["Xbox"] = "Xbox";
})(IdentityProviderType = exports.IdentityProviderType || (exports.IdentityProviderType = {}));
/**
 * Models the reason that {@link UserIdentity#verified} was set to true or false.
 *
 * @author Brady Wied
 */
var IdentityVerifiedReason;
(function (IdentityVerifiedReason) {
    IdentityVerifiedReason["Skipped"] = "Skipped";
    IdentityVerifiedReason["Trusted"] = "Trusted";
    IdentityVerifiedReason["Unverifiable"] = "Unverifiable";
    IdentityVerifiedReason["Implicit"] = "Implicit";
    IdentityVerifiedReason["Pending"] = "Pending";
    IdentityVerifiedReason["Completed"] = "Completed";
    IdentityVerifiedReason["Disabled"] = "Disabled";
    IdentityVerifiedReason["Administrative"] = "Administrative";
    IdentityVerifiedReason["Import"] = "Import";
})(IdentityVerifiedReason = exports.IdentityVerifiedReason || (exports.IdentityVerifiedReason = {}));
var KeyAlgorithm;
(function (KeyAlgorithm) {
    KeyAlgorithm["ES256"] = "ES256";
    KeyAlgorithm["ES384"] = "ES384";
    KeyAlgorithm["ES512"] = "ES512";
    KeyAlgorithm["HS256"] = "HS256";
    KeyAlgorithm["HS384"] = "HS384";
    KeyAlgorithm["HS512"] = "HS512";
    KeyAlgorithm["RS256"] = "RS256";
    KeyAlgorithm["RS384"] = "RS384";
    KeyAlgorithm["RS512"] = "RS512";
})(KeyAlgorithm = exports.KeyAlgorithm || (exports.KeyAlgorithm = {}));
var KeyType;
(function (KeyType) {
    KeyType["EC"] = "EC";
    KeyType["RSA"] = "RSA";
    KeyType["HMAC"] = "HMAC";
})(KeyType = exports.KeyType || (exports.KeyType = {}));
/**
 * The use type of a key.
 *
 * @author Daniel DeGroff
 */
var KeyUse;
(function (KeyUse) {
    KeyUse["SignOnly"] = "SignOnly";
    KeyUse["SignAndVerify"] = "SignAndVerify";
    KeyUse["VerifyOnly"] = "VerifyOnly";
})(KeyUse = exports.KeyUse || (exports.KeyUse = {}));
var LDAPSecurityMethod;
(function (LDAPSecurityMethod) {
    LDAPSecurityMethod["None"] = "None";
    LDAPSecurityMethod["LDAPS"] = "LDAPS";
    LDAPSecurityMethod["StartTLS"] = "StartTLS";
})(LDAPSecurityMethod = exports.LDAPSecurityMethod || (exports.LDAPSecurityMethod = {}));
/**
 * @author Daniel DeGroff
 */
var LambdaEngineType;
(function (LambdaEngineType) {
    LambdaEngineType["GraalJS"] = "GraalJS";
    LambdaEngineType["Nashorn"] = "Nashorn";
})(LambdaEngineType = exports.LambdaEngineType || (exports.LambdaEngineType = {}));
/**
 * The types of lambdas that indicate how they are invoked by FusionAuth.
 *
 * @author Brian Pontarelli
 */
var LambdaType;
(function (LambdaType) {
    LambdaType["JWTPopulate"] = "JWTPopulate";
    LambdaType["OpenIDReconcile"] = "OpenIDReconcile";
    LambdaType["SAMLv2Reconcile"] = "SAMLv2Reconcile";
    LambdaType["SAMLv2Populate"] = "SAMLv2Populate";
    LambdaType["AppleReconcile"] = "AppleReconcile";
    LambdaType["ExternalJWTReconcile"] = "ExternalJWTReconcile";
    LambdaType["FacebookReconcile"] = "FacebookReconcile";
    LambdaType["GoogleReconcile"] = "GoogleReconcile";
    LambdaType["HYPRReconcile"] = "HYPRReconcile";
    LambdaType["TwitterReconcile"] = "TwitterReconcile";
    LambdaType["LDAPConnectorReconcile"] = "LDAPConnectorReconcile";
    LambdaType["LinkedInReconcile"] = "LinkedInReconcile";
    LambdaType["EpicGamesReconcile"] = "EpicGamesReconcile";
    LambdaType["NintendoReconcile"] = "NintendoReconcile";
    LambdaType["SonyPSNReconcile"] = "SonyPSNReconcile";
    LambdaType["SteamReconcile"] = "SteamReconcile";
    LambdaType["TwitchReconcile"] = "TwitchReconcile";
    LambdaType["XboxReconcile"] = "XboxReconcile";
    LambdaType["ClientCredentialsJWTPopulate"] = "ClientCredentialsJWTPopulate";
    LambdaType["SCIMServerGroupRequestConverter"] = "SCIMServerGroupRequestConverter";
    LambdaType["SCIMServerGroupResponseConverter"] = "SCIMServerGroupResponseConverter";
    LambdaType["SCIMServerUserRequestConverter"] = "SCIMServerUserRequestConverter";
    LambdaType["SCIMServerUserResponseConverter"] = "SCIMServerUserResponseConverter";
    LambdaType["SelfServiceRegistrationValidation"] = "SelfServiceRegistrationValidation";
    LambdaType["UserInfoPopulate"] = "UserInfoPopulate";
    LambdaType["LoginValidation"] = "LoginValidation";
})(LambdaType = exports.LambdaType || (exports.LambdaType = {}));
/**
 * @author Matthew Altman
 */
var LogoutBehavior;
(function (LogoutBehavior) {
    LogoutBehavior["RedirectOnly"] = "RedirectOnly";
    LogoutBehavior["AllApplications"] = "AllApplications";
})(LogoutBehavior = exports.LogoutBehavior || (exports.LogoutBehavior = {}));
/**
 * @author Mikey Sleevi
 */
var MessageType;
(function (MessageType) {
    MessageType["SMS"] = "SMS";
})(MessageType = exports.MessageType || (exports.MessageType = {}));
/**
 * @author Brett Guy
 */
var MessengerType;
(function (MessengerType) {
    MessengerType["Generic"] = "Generic";
    MessengerType["Kafka"] = "Kafka";
    MessengerType["Twilio"] = "Twilio";
})(MessengerType = exports.MessengerType || (exports.MessengerType = {}));
/**
 * @author Daniel DeGroff
 */
var MultiFactorLoginPolicy;
(function (MultiFactorLoginPolicy) {
    MultiFactorLoginPolicy["Disabled"] = "Disabled";
    MultiFactorLoginPolicy["Enabled"] = "Enabled";
    MultiFactorLoginPolicy["Required"] = "Required";
})(MultiFactorLoginPolicy = exports.MultiFactorLoginPolicy || (exports.MultiFactorLoginPolicy = {}));
/**
 * The application's relationship to the authorization server. First-party applications will be granted implicit permission for requested scopes.
 * Third-party applications will use the {@link OAuthScopeConsentMode} policy.
 *
 * @author Spencer Witt
 */
var OAuthApplicationRelationship;
(function (OAuthApplicationRelationship) {
    OAuthApplicationRelationship["FirstParty"] = "FirstParty";
    OAuthApplicationRelationship["ThirdParty"] = "ThirdParty";
})(OAuthApplicationRelationship = exports.OAuthApplicationRelationship || (exports.OAuthApplicationRelationship = {}));
var OAuthErrorReason;
(function (OAuthErrorReason) {
    OAuthErrorReason["auth_code_not_found"] = "auth_code_not_found";
    OAuthErrorReason["access_token_malformed"] = "access_token_malformed";
    OAuthErrorReason["access_token_expired"] = "access_token_expired";
    OAuthErrorReason["access_token_unavailable_for_processing"] = "access_token_unavailable_for_processing";
    OAuthErrorReason["access_token_failed_processing"] = "access_token_failed_processing";
    OAuthErrorReason["access_token_invalid"] = "access_token_invalid";
    OAuthErrorReason["access_token_required"] = "access_token_required";
    OAuthErrorReason["refresh_token_not_found"] = "refresh_token_not_found";
    OAuthErrorReason["refresh_token_type_not_supported"] = "refresh_token_type_not_supported";
    OAuthErrorReason["id_token_invalid"] = "id_token_invalid";
    OAuthErrorReason["unsupported_token_type"] = "unsupported_token_type";
    OAuthErrorReason["token_type_hint_mismatch"] = "token_type_hint_mismatch";
    OAuthErrorReason["invalid_client_id"] = "invalid_client_id";
    OAuthErrorReason["invalid_expires_in"] = "invalid_expires_in";
    OAuthErrorReason["invalid_user_credentials"] = "invalid_user_credentials";
    OAuthErrorReason["invalid_grant_type"] = "invalid_grant_type";
    OAuthErrorReason["invalid_origin"] = "invalid_origin";
    OAuthErrorReason["invalid_origin_opaque"] = "invalid_origin_opaque";
    OAuthErrorReason["invalid_pkce_code_verifier"] = "invalid_pkce_code_verifier";
    OAuthErrorReason["invalid_pkce_code_challenge"] = "invalid_pkce_code_challenge";
    OAuthErrorReason["invalid_pkce_code_challenge_method"] = "invalid_pkce_code_challenge_method";
    OAuthErrorReason["invalid_prompt"] = "invalid_prompt";
    OAuthErrorReason["invalid_redirect_uri"] = "invalid_redirect_uri";
    OAuthErrorReason["invalid_response_mode"] = "invalid_response_mode";
    OAuthErrorReason["invalid_response_type"] = "invalid_response_type";
    OAuthErrorReason["invalid_id_token_hint"] = "invalid_id_token_hint";
    OAuthErrorReason["invalid_post_logout_redirect_uri"] = "invalid_post_logout_redirect_uri";
    OAuthErrorReason["invalid_device_code"] = "invalid_device_code";
    OAuthErrorReason["invalid_user_code"] = "invalid_user_code";
    OAuthErrorReason["invalid_additional_client_id"] = "invalid_additional_client_id";
    OAuthErrorReason["invalid_target_entity_scope"] = "invalid_target_entity_scope";
    OAuthErrorReason["invalid_entity_permission_scope"] = "invalid_entity_permission_scope";
    OAuthErrorReason["invalid_user_id"] = "invalid_user_id";
    OAuthErrorReason["invalid_tenant_id"] = "invalid_tenant_id";
    OAuthErrorReason["grant_type_disabled"] = "grant_type_disabled";
    OAuthErrorReason["missing_client_id"] = "missing_client_id";
    OAuthErrorReason["missing_client_secret"] = "missing_client_secret";
    OAuthErrorReason["missing_code"] = "missing_code";
    OAuthErrorReason["missing_code_challenge"] = "missing_code_challenge";
    OAuthErrorReason["missing_code_verifier"] = "missing_code_verifier";
    OAuthErrorReason["missing_device_code"] = "missing_device_code";
    OAuthErrorReason["missing_grant_type"] = "missing_grant_type";
    OAuthErrorReason["missing_redirect_uri"] = "missing_redirect_uri";
    OAuthErrorReason["missing_refresh_token"] = "missing_refresh_token";
    OAuthErrorReason["missing_response_type"] = "missing_response_type";
    OAuthErrorReason["missing_token"] = "missing_token";
    OAuthErrorReason["missing_user_code"] = "missing_user_code";
    OAuthErrorReason["missing_user_id"] = "missing_user_id";
    OAuthErrorReason["missing_verification_uri"] = "missing_verification_uri";
    OAuthErrorReason["missing_tenant_id"] = "missing_tenant_id";
    OAuthErrorReason["login_prevented"] = "login_prevented";
    OAuthErrorReason["not_licensed"] = "not_licensed";
    OAuthErrorReason["user_code_expired"] = "user_code_expired";
    OAuthErrorReason["user_expired"] = "user_expired";
    OAuthErrorReason["user_locked"] = "user_locked";
    OAuthErrorReason["user_not_found"] = "user_not_found";
    OAuthErrorReason["client_authentication_missing"] = "client_authentication_missing";
    OAuthErrorReason["invalid_client_authentication_scheme"] = "invalid_client_authentication_scheme";
    OAuthErrorReason["invalid_client_authentication"] = "invalid_client_authentication";
    OAuthErrorReason["client_id_mismatch"] = "client_id_mismatch";
    OAuthErrorReason["change_password_administrative"] = "change_password_administrative";
    OAuthErrorReason["change_password_breached"] = "change_password_breached";
    OAuthErrorReason["change_password_expired"] = "change_password_expired";
    OAuthErrorReason["change_password_validation"] = "change_password_validation";
    OAuthErrorReason["unknown"] = "unknown";
    OAuthErrorReason["missing_required_scope"] = "missing_required_scope";
    OAuthErrorReason["unknown_scope"] = "unknown_scope";
    OAuthErrorReason["consent_canceled"] = "consent_canceled";
    OAuthErrorReason["authentication_required"] = "authentication_required";
    OAuthErrorReason["email_verification_required"] = "email_verification_required";
    OAuthErrorReason["multi_factor_challenge_required"] = "multi_factor_challenge_required";
    OAuthErrorReason["phone_verification_required"] = "phone_verification_required";
    OAuthErrorReason["registration_missing_requirement"] = "registration_missing_requirement";
    OAuthErrorReason["registration_required"] = "registration_required";
    OAuthErrorReason["registration_verification_required"] = "registration_verification_required";
    OAuthErrorReason["consent_required"] = "consent_required";
})(OAuthErrorReason = exports.OAuthErrorReason || (exports.OAuthErrorReason = {}));
var OAuthErrorType;
(function (OAuthErrorType) {
    OAuthErrorType["invalid_request"] = "invalid_request";
    OAuthErrorType["invalid_client"] = "invalid_client";
    OAuthErrorType["invalid_grant"] = "invalid_grant";
    OAuthErrorType["invalid_token"] = "invalid_token";
    OAuthErrorType["consent_required"] = "consent_required";
    OAuthErrorType["interaction_required"] = "interaction_required";
    OAuthErrorType["login_required"] = "login_required";
    OAuthErrorType["unauthorized_client"] = "unauthorized_client";
    OAuthErrorType["invalid_scope"] = "invalid_scope";
    OAuthErrorType["server_error"] = "server_error";
    OAuthErrorType["unsupported_grant_type"] = "unsupported_grant_type";
    OAuthErrorType["unsupported_response_type"] = "unsupported_response_type";
    OAuthErrorType["access_denied"] = "access_denied";
    OAuthErrorType["change_password_required"] = "change_password_required";
    OAuthErrorType["not_licensed"] = "not_licensed";
    OAuthErrorType["two_factor_required"] = "two_factor_required";
    OAuthErrorType["authorization_pending"] = "authorization_pending";
    OAuthErrorType["expired_token"] = "expired_token";
    OAuthErrorType["unsupported_token_type"] = "unsupported_token_type";
})(OAuthErrorType = exports.OAuthErrorType || (exports.OAuthErrorType = {}));
/**
 * Controls the policy for requesting user permission to grant access to requested scopes during an OAuth workflow
 * for a third-party application.
 *
 * @author Spencer Witt
 */
var OAuthScopeConsentMode;
(function (OAuthScopeConsentMode) {
    OAuthScopeConsentMode["AlwaysPrompt"] = "AlwaysPrompt";
    OAuthScopeConsentMode["RememberDecision"] = "RememberDecision";
    OAuthScopeConsentMode["NeverPrompt"] = "NeverPrompt";
})(OAuthScopeConsentMode = exports.OAuthScopeConsentMode || (exports.OAuthScopeConsentMode = {}));
/**
 * Controls the policy for whether OAuth workflows will more strictly adhere to the OAuth and OIDC specification
 * or run in backwards compatibility mode.
 *
 * @author David Charles
 */
var OAuthScopeHandlingPolicy;
(function (OAuthScopeHandlingPolicy) {
    OAuthScopeHandlingPolicy["Compatibility"] = "Compatibility";
    OAuthScopeHandlingPolicy["Strict"] = "Strict";
})(OAuthScopeHandlingPolicy = exports.OAuthScopeHandlingPolicy || (exports.OAuthScopeHandlingPolicy = {}));
/**
 * @author Johnathon Wood
 */
var Oauth2AuthorizedURLValidationPolicy;
(function (Oauth2AuthorizedURLValidationPolicy) {
    Oauth2AuthorizedURLValidationPolicy["AllowWildcards"] = "AllowWildcards";
    Oauth2AuthorizedURLValidationPolicy["ExactMatch"] = "ExactMatch";
})(Oauth2AuthorizedURLValidationPolicy = exports.Oauth2AuthorizedURLValidationPolicy || (exports.Oauth2AuthorizedURLValidationPolicy = {}));
/**
 * @author Daniel DeGroff
 */
var ObjectState;
(function (ObjectState) {
    ObjectState["Active"] = "Active";
    ObjectState["Inactive"] = "Inactive";
    ObjectState["PendingDelete"] = "PendingDelete";
})(ObjectState = exports.ObjectState || (exports.ObjectState = {}));
var BreachAction;
(function (BreachAction) {
    BreachAction["Off"] = "Off";
    BreachAction["RecordOnly"] = "RecordOnly";
    BreachAction["NotifyUser"] = "NotifyUser";
    BreachAction["RequireChange"] = "RequireChange";
})(BreachAction = exports.BreachAction || (exports.BreachAction = {}));
var BreachMatchMode;
(function (BreachMatchMode) {
    BreachMatchMode["Low"] = "Low";
    BreachMatchMode["Medium"] = "Medium";
    BreachMatchMode["High"] = "High";
})(BreachMatchMode = exports.BreachMatchMode || (exports.BreachMatchMode = {}));
/**
 * @author Daniel DeGroff
 */
var PasswordlessStrategy;
(function (PasswordlessStrategy) {
    PasswordlessStrategy["ClickableLink"] = "ClickableLink";
    PasswordlessStrategy["FormField"] = "FormField";
})(PasswordlessStrategy = exports.PasswordlessStrategy || (exports.PasswordlessStrategy = {}));
/**
 * @author Brett Guy
 */
var ProofKeyForCodeExchangePolicy;
(function (ProofKeyForCodeExchangePolicy) {
    ProofKeyForCodeExchangePolicy["Required"] = "Required";
    ProofKeyForCodeExchangePolicy["NotRequired"] = "NotRequired";
    ProofKeyForCodeExchangePolicy["NotRequiredWhenUsingClientAuthentication"] = "NotRequiredWhenUsingClientAuthentication";
})(ProofKeyForCodeExchangePolicy = exports.ProofKeyForCodeExchangePolicy || (exports.ProofKeyForCodeExchangePolicy = {}));
/**
 * Defines valid credential types. This is an extension point in the WebAuthn spec. The only defined value at this time is "public-key"
 *
 * @author Spencer Witt
 */
var PublicKeyCredentialType;
(function (PublicKeyCredentialType) {
    PublicKeyCredentialType["publicKey"] = "public-key";
})(PublicKeyCredentialType = exports.PublicKeyCredentialType || (exports.PublicKeyCredentialType = {}));
/**
 * @author Brian Pontarelli
 */
var ReactorFeatureStatus;
(function (ReactorFeatureStatus) {
    ReactorFeatureStatus["ACTIVE"] = "ACTIVE";
    ReactorFeatureStatus["DISCONNECTED"] = "DISCONNECTED";
    ReactorFeatureStatus["PENDING"] = "PENDING";
    ReactorFeatureStatus["DISABLED"] = "DISABLED";
    ReactorFeatureStatus["UNKNOWN"] = "UNKNOWN";
})(ReactorFeatureStatus = exports.ReactorFeatureStatus || (exports.ReactorFeatureStatus = {}));
/**
 * @author Daniel DeGroff
 */
var RefreshTokenExpirationPolicy;
(function (RefreshTokenExpirationPolicy) {
    RefreshTokenExpirationPolicy["Fixed"] = "Fixed";
    RefreshTokenExpirationPolicy["SlidingWindow"] = "SlidingWindow";
    RefreshTokenExpirationPolicy["SlidingWindowWithMaximumLifetime"] = "SlidingWindowWithMaximumLifetime";
})(RefreshTokenExpirationPolicy = exports.RefreshTokenExpirationPolicy || (exports.RefreshTokenExpirationPolicy = {}));
/**
 * @author Daniel DeGroff
 */
var RefreshTokenUsagePolicy;
(function (RefreshTokenUsagePolicy) {
    RefreshTokenUsagePolicy["Reusable"] = "Reusable";
    RefreshTokenUsagePolicy["OneTimeUse"] = "OneTimeUse";
})(RefreshTokenUsagePolicy = exports.RefreshTokenUsagePolicy || (exports.RefreshTokenUsagePolicy = {}));
/**
 * Describes the Relying Party's requirements for <a href="https://www.w3.org/TR/webauthn-2/#client-side-discoverable-credential">client-side
 * discoverable credentials</a> (formerly known as "resident keys")
 *
 * @author Spencer Witt
 */
var ResidentKeyRequirement;
(function (ResidentKeyRequirement) {
    ResidentKeyRequirement["discouraged"] = "discouraged";
    ResidentKeyRequirement["preferred"] = "preferred";
    ResidentKeyRequirement["required"] = "required";
})(ResidentKeyRequirement = exports.ResidentKeyRequirement || (exports.ResidentKeyRequirement = {}));
/**
 * @author Lyle Schemmerling
 */
var SAMLv2DestinationAssertionPolicy;
(function (SAMLv2DestinationAssertionPolicy) {
    SAMLv2DestinationAssertionPolicy["Enabled"] = "Enabled";
    SAMLv2DestinationAssertionPolicy["Disabled"] = "Disabled";
    SAMLv2DestinationAssertionPolicy["AllowAlternates"] = "AllowAlternates";
})(SAMLv2DestinationAssertionPolicy = exports.SAMLv2DestinationAssertionPolicy || (exports.SAMLv2DestinationAssertionPolicy = {}));
/**
 * @author Daniel DeGroff
 */
var SecureGeneratorType;
(function (SecureGeneratorType) {
    SecureGeneratorType["randomDigits"] = "randomDigits";
    SecureGeneratorType["randomBytes"] = "randomBytes";
    SecureGeneratorType["randomAlpha"] = "randomAlpha";
    SecureGeneratorType["randomAlphaNumeric"] = "randomAlphaNumeric";
})(SecureGeneratorType = exports.SecureGeneratorType || (exports.SecureGeneratorType = {}));
/**
 * Used to indicate which identity type a password "request" might go to. It could be
 * used for send set passwords or send password resets.
 */
var SendSetPasswordIdentityType;
(function (SendSetPasswordIdentityType) {
    SendSetPasswordIdentityType["email"] = "email";
    SendSetPasswordIdentityType["phone"] = "phone";
    SendSetPasswordIdentityType["doNotSend"] = "doNotSend";
})(SendSetPasswordIdentityType = exports.SendSetPasswordIdentityType || (exports.SendSetPasswordIdentityType = {}));
/**
 * @author Daniel DeGroff
 */
var Sort;
(function (Sort) {
    Sort["asc"] = "asc";
    Sort["desc"] = "desc";
})(Sort = exports.Sort || (exports.Sort = {}));
/**
 * Steam API modes.
 *
 * @author Daniel DeGroff
 */
var SteamAPIMode;
(function (SteamAPIMode) {
    SteamAPIMode["Public"] = "Public";
    SteamAPIMode["Partner"] = "Partner";
})(SteamAPIMode = exports.SteamAPIMode || (exports.SteamAPIMode = {}));
/**
 * @author Daniel DeGroff
 */
var SystemTrustedProxyConfigurationPolicy;
(function (SystemTrustedProxyConfigurationPolicy) {
    SystemTrustedProxyConfigurationPolicy["All"] = "All";
    SystemTrustedProxyConfigurationPolicy["OnlyConfigured"] = "OnlyConfigured";
})(SystemTrustedProxyConfigurationPolicy = exports.SystemTrustedProxyConfigurationPolicy || (exports.SystemTrustedProxyConfigurationPolicy = {}));
var UniqueUsernameStrategy;
(function (UniqueUsernameStrategy) {
    UniqueUsernameStrategy["Always"] = "Always";
    UniqueUsernameStrategy["OnCollision"] = "OnCollision";
})(UniqueUsernameStrategy = exports.UniqueUsernameStrategy || (exports.UniqueUsernameStrategy = {}));
var ThemeType;
(function (ThemeType) {
    ThemeType["advanced"] = "advanced";
    ThemeType["simple"] = "simple";
})(ThemeType = exports.ThemeType || (exports.ThemeType = {}));
/**
 * <ul>
 * <li>Bearer Token type as defined by <a href="https://tools.ietf.org/html/rfc6750">RFC 6750</a>.</li>
 * <li>MAC Token type as referenced by <a href="https://tools.ietf.org/html/rfc6749">RFC 6749</a> and
 * <a href="https://tools.ietf.org/html/draft-ietf-oauth-v2-http-mac-05">
 * Draft RFC on OAuth 2.0 Message Authentication Code (MAC) Tokens</a>
 * </li>
 * </ul>
 *
 * @author Daniel DeGroff
 */
var TokenType;
(function (TokenType) {
    TokenType["Bearer"] = "Bearer";
    TokenType["MAC"] = "MAC";
})(TokenType = exports.TokenType || (exports.TokenType = {}));
/**
 * The transaction types for Webhooks and other event systems within FusionAuth.
 *
 * @author Brian Pontarelli
 */
var TransactionType;
(function (TransactionType) {
    TransactionType["None"] = "None";
    TransactionType["Any"] = "Any";
    TransactionType["SimpleMajority"] = "SimpleMajority";
    TransactionType["SuperMajority"] = "SuperMajority";
    TransactionType["AbsoluteMajority"] = "AbsoluteMajority";
})(TransactionType = exports.TransactionType || (exports.TransactionType = {}));
/**
 * Policy for handling unknown OAuth scopes in the request
 *
 * @author Spencer Witt
 */
var UnknownScopePolicy;
(function (UnknownScopePolicy) {
    UnknownScopePolicy["Allow"] = "Allow";
    UnknownScopePolicy["Remove"] = "Remove";
    UnknownScopePolicy["Reject"] = "Reject";
})(UnknownScopePolicy = exports.UnknownScopePolicy || (exports.UnknownScopePolicy = {}));
/**
 * @author Daniel DeGroff
 */
var UnverifiedBehavior;
(function (UnverifiedBehavior) {
    UnverifiedBehavior["Allow"] = "Allow";
    UnverifiedBehavior["Gated"] = "Gated";
})(UnverifiedBehavior = exports.UnverifiedBehavior || (exports.UnverifiedBehavior = {}));
/**
 * The phases of a time-based user action.
 *
 * @author Brian Pontarelli
 */
var UserActionPhase;
(function (UserActionPhase) {
    UserActionPhase["start"] = "start";
    UserActionPhase["modify"] = "modify";
    UserActionPhase["cancel"] = "cancel";
    UserActionPhase["end"] = "end";
})(UserActionPhase = exports.UserActionPhase || (exports.UserActionPhase = {}));
/**
 * @author Daniel DeGroff
 */
var UserState;
(function (UserState) {
    UserState["Authenticated"] = "Authenticated";
    UserState["AuthenticatedNotRegistered"] = "AuthenticatedNotRegistered";
    UserState["AuthenticatedNotVerified"] = "AuthenticatedNotVerified";
    UserState["AuthenticatedRegistrationNotVerified"] = "AuthenticatedRegistrationNotVerified";
})(UserState = exports.UserState || (exports.UserState = {}));
/**
 * Used to express whether the Relying Party requires <a href="https://www.w3.org/TR/webauthn-2/#user-verification">user verification</a> for the
 * current operation.
 *
 * @author Spencer Witt
 */
var UserVerificationRequirement;
(function (UserVerificationRequirement) {
    UserVerificationRequirement["required"] = "required";
    UserVerificationRequirement["preferred"] = "preferred";
    UserVerificationRequirement["discouraged"] = "discouraged";
})(UserVerificationRequirement = exports.UserVerificationRequirement || (exports.UserVerificationRequirement = {}));
/**
 * @author Daniel DeGroff
 */
var VerificationStrategy;
(function (VerificationStrategy) {
    VerificationStrategy["ClickableLink"] = "ClickableLink";
    VerificationStrategy["FormField"] = "FormField";
})(VerificationStrategy = exports.VerificationStrategy || (exports.VerificationStrategy = {}));
/**
 * Identifies the WebAuthn workflow. This will affect the parameters used for credential creation
 * and request based on the Tenant configuration.
 *
 * @author Spencer Witt
 */
var WebAuthnWorkflow;
(function (WebAuthnWorkflow) {
    WebAuthnWorkflow["bootstrap"] = "bootstrap";
    WebAuthnWorkflow["general"] = "general";
    WebAuthnWorkflow["reauthentication"] = "reauthentication";
})(WebAuthnWorkflow = exports.WebAuthnWorkflow || (exports.WebAuthnWorkflow = {}));
/**
 * The possible states of an individual webhook attempt to a single endpoint.
 *
 * @author Spencer Witt
 */
var WebhookAttemptResult;
(function (WebhookAttemptResult) {
    WebhookAttemptResult["Success"] = "Success";
    WebhookAttemptResult["Failure"] = "Failure";
    WebhookAttemptResult["Unknown"] = "Unknown";
})(WebhookAttemptResult = exports.WebhookAttemptResult || (exports.WebhookAttemptResult = {}));
/**
 * The possible result states of a webhook event. This tracks the success of the overall webhook transaction according to the {@link TransactionType}
 * and configured webhooks.
 *
 * @author Spencer Witt
 */
var WebhookEventResult;
(function (WebhookEventResult) {
    WebhookEventResult["Failed"] = "Failed";
    WebhookEventResult["Running"] = "Running";
    WebhookEventResult["Succeeded"] = "Succeeded";
})(WebhookEventResult = exports.WebhookEventResult || (exports.WebhookEventResult = {}));
//# sourceMappingURL=FusionAuthClient.js.map