/*
 * Copyright (c) 2022, FusionAuth, All Rights Reserved
 */
package io.fusionauth.domain;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

import com.inversoft.json.JacksonConstructor;
import com.inversoft.json.ToString;
import io.fusionauth.domain.webauthn.AttestationType;
import io.fusionauth.domain.webauthn.CoseAlgorithmIdentifier;

/**
 * A User's WebAuthnCredential. Contains all data required to complete WebAuthn authentication ceremonies.
 *
 * @author Spencer Witt
 */
public class WebAuthnCredential implements Tenantable, Buildable<WebAuthnCredential> {
  /**
   * The signature algorithm used with the key
   */
  public CoseAlgorithmIdentifier algorithm;

  /**
   * The attestation type that was provided during credential registration
   */
  public AttestationType attestationType;

  /**
   * Indicates whether user verification is supported by the generating authenticator based on whether user verification was employed during
   * registration
   */
  public boolean authenticatorSupportsUserVerification;

  /**
   * The credential ID generated by the <i>authenticator</i> stored as a base64URL-encoded String
   */
  public String credentialId;

  public Map<String, Object> data = new LinkedHashMap<>();

  /**
   * Indicates whether the credential is a client-side discoverable credential. If the authenticator could not provide assurances one way or the
   * other, the value is <code>false</code>. Requires the WebAuthn <code>credProps</code> extension
   */
  public boolean discoverable;

  /**
   * The display name selected during credential registration. This is a user-supplied value that defaults to their loginId.
   */
  public String displayName;

  public UUID id;

  /**
   * Timestamp for credential creation
   */
  public ZonedDateTime insertInstant;

  /**
   * Timestamp for the last time the credential was used for an authentication ceremony
   */
  public ZonedDateTime lastUseInstant;

  /**
   * The unique name assigned during credential registration. This value is meant to disambiguate credentials with the same
   * {@link WebAuthnCredential#displayName}
   */
  public String name;

  /**
   * The public key encoded in PEM format
   */
  public String publicKey;

  /**
   * The Relying Party Id used when the credential was registered
   */
  public String relyingPartyId;

  /**
   * The number of signatures generated with the key
   */
  public int signCount;

  public UUID tenantId;

  /**
   * A list of transport types supported by the WebAuthn <i>authenticator</i> that generated the key
   */
  public List<String> transports = new ArrayList<>();

  /**
   * The user agent string from the browser that registered the credential
   */
  public String userAgent;

  public UUID userId;

  @JacksonConstructor
  public WebAuthnCredential() {
  }

  public WebAuthnCredential(WebAuthnCredential other) {
    this.algorithm = other.algorithm;
    this.attestationType = other.attestationType;
    this.authenticatorSupportsUserVerification = other.authenticatorSupportsUserVerification;
    this.credentialId = other.credentialId;
    if (other.data != null) {
      this.data.putAll(other.data);
    }
    this.discoverable = other.discoverable;
    this.displayName = other.displayName;
    this.id = other.id;
    this.insertInstant = other.insertInstant;
    this.name = other.name;
    this.lastUseInstant = other.lastUseInstant;
    this.publicKey = other.publicKey;
    this.relyingPartyId = other.relyingPartyId;
    this.signCount = other.signCount;
    this.tenantId = other.tenantId;
    if (other.transports != null) {
      this.transports.addAll(other.transports);
    }
    this.userAgent = other.userAgent;
    this.userId = other.userId;

  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    WebAuthnCredential that = (WebAuthnCredential) o;
    return authenticatorSupportsUserVerification == that.authenticatorSupportsUserVerification &&
           discoverable == that.discoverable &&
           signCount == that.signCount &&
           algorithm == that.algorithm &&
           attestationType == that.attestationType &&
           Objects.equals(credentialId, that.credentialId) &&
           Objects.equals(data, that.data) &&
           Objects.equals(displayName, that.displayName) &&
           Objects.equals(id, that.id) &&
           Objects.equals(insertInstant, that.insertInstant) &&
           Objects.equals(lastUseInstant, that.lastUseInstant) &&
           Objects.equals(name, that.name) &&
           Objects.equals(publicKey, that.publicKey) &&
           Objects.equals(relyingPartyId, that.relyingPartyId) &&
           Objects.equals(tenantId, that.tenantId) &&
           Objects.equals(transports, that.transports) &&
           Objects.equals(userAgent, that.userAgent) &&
           Objects.equals(userId, that.userId);
  }

  @Override
  public UUID getTenantId() {
    return tenantId;
  }

  @Override
  public int hashCode() {
    return Objects.hash(algorithm,
                        attestationType,
                        authenticatorSupportsUserVerification,
                        credentialId,
                        data,
                        discoverable,
                        displayName,
                        id,
                        insertInstant,
                        lastUseInstant,
                        name,
                        publicKey,
                        relyingPartyId,
                        signCount,
                        tenantId,
                        transports,
                        userAgent,
                        userId);
  }

  @Override
  public String toString() {
    return ToString.toString(this);
  }
}
